<?php
// =======================================================
// ARQUIVO DE API DO DASHBOARD (VERSÃO ATUALIZADA E CORRIGIDA)
// =======================================================

// Define o cabeçalho para a resposta ser sempre em formato JSON
header('Content-Type: application/json; charset=UTF-8');
session_start(); // Essencial para saber quem está logado

// -------------------------------------------------------------------
// PASSO 1: VERIFICAÇÃO DE SEGURANÇA
// -------------------------------------------------------------------
// Verifica se o usuário (admin/revendedor) está logado na sessão.
if (!isset($_SESSION['logged_in_fxtream']) || $_SESSION['logged_in_fxtream'] !== true) {
    http_response_code(403); // Código de "Acesso Proibido"
    echo json_encode(['error' => 'Acesso não autorizado. Faça o login no painel.']);
    exit;
}

// -------------------------------------------------------------------
// PASSO 2: CONEXÃO COM O BANCO DE DADOS
// -------------------------------------------------------------------
// Certifique-se que o caminho para 'controles/db.php' está correto.
// O caminho deve ser relativo à localização deste arquivo de API.
require_once 'controles/db.php';
date_default_timezone_set('America/Sao_Paulo');

// Pega os dados do usuário logado a partir da sessão
$loggedInUserId = $_SESSION['admin_id'];
$userLevel = $_SESSION['nivel_admin'];

try {
    $pdo = conectar_bd();

    // -------------------------------------------------------------------
    // PASSO 3: LÓGICA DE "SEMPRE VISÍVEL" (MUDANÇA PRINCIPAL AQUI)
    // -------------------------------------------------------------------
    // Limpa apenas conexões que estão realmente inativas há mais de 3 minutos.
    // Isso garante que clientes assistindo NUNCA desapareçam da lista.
    $pdo->exec("DELETE FROM conexoes WHERE ultima_atividade < NOW() - INTERVAL 180 SECOND");

    // --- Montagem dinâmica da query com base no nível do usuário ---
    $reseller_filter = '';
    $params = [];
    if ($userLevel != 1) { // Se não for admin principal, aplica o filtro
        $reseller_filter = " AND c.admin_id = :admin_id ";
        $params[':admin_id'] = $loggedInUserId;
    }

    // -------------------------------------------------------------------
    // PASSO 4: BUSCAR DADOS DO BANCO
    // -------------------------------------------------------------------

    // --- QUERY DETALHADA: Lista todas as conexões ativas ---
    // Esta query busca os dados para a tabela de "Log de Conexões"
    $sql_detalhado = "
        SELECT 
            co.usuario, co.ip, co.ultima_atividade,
            COALESCE(s.name, se.title, 'Conteúdo Desconhecido') AS stream_name,
            IF(s.id IS NOT NULL, s.stream_type, 'series') AS stream_type
        FROM conexoes co
        JOIN clientes c ON co.usuario = c.usuario
        LEFT JOIN streams s ON co.stream_id = s.id
        LEFT JOIN series_episodes se ON co.stream_id = se.id AND s.id IS NULL
        WHERE 1=1 {$reseller_filter}
        ORDER BY co.ultima_atividade DESC;
    ";
    $stmt_detalhado = $pdo->prepare($sql_detalhado);
    $stmt_detalhado->execute($params);
    $conexoes_detalhadas = $stmt_detalhado->fetchAll(PDO::FETCH_ASSOC);

    // --- QUERY AGREGADA: Lista os conteúdos mais assistidos (mantivemos sua ótima query) ---
    // Esta query busca os dados para a lista de "Top Conteúdos"
    $sql_agregado = "
        SELECT 
            T.name, T.stream_icon, COUNT(T.conexao_id) as total_espectadores
        FROM (
            SELECT s.name, s.stream_icon, c.id as conexao_id
            FROM conexoes c
            JOIN clientes cl ON c.usuario = cl.usuario
            JOIN streams s ON c.stream_id = s.id
            WHERE c.stream_id IS NOT NULL {$reseller_filter}
            UNION ALL
            SELECT se.title as name, se.movie_image as stream_icon, c.id as conexao_id
            FROM conexoes c
            JOIN clientes cl ON c.usuario = cl.usuario
            JOIN series_episodes se ON c.stream_id = se.id
            WHERE 1=1 {$reseller_filter}
        ) AS T
        GROUP BY T.name, T.stream_icon
        ORDER BY total_espectadores DESC, T.name ASC
        LIMIT 5;
    ";
    $stmt_agregado = $pdo->prepare($sql_agregado);
    $stmt_agregado->execute($params);
    $conteudo_popular = $stmt_agregado->fetchAll(PDO::FETCH_ASSOC);


    // -------------------------------------------------------------------
    // PASSO 5: PROCESSAR E FORMATAR OS DADOS PARA JSON
    // -------------------------------------------------------------------
    function formatar_tempo_passado_api($datetime_str) {
        if(!$datetime_str) return 'N/A';
        try {
            $now = new DateTime(); $ago = new DateTime($datetime_str); $diff = $now->diff($ago);
            if ($diff->days > 0) return 'há ' . $diff->days . 'd';
            if ($diff->h > 0) return 'há ' . $diff->h . 'h';
            if ($diff->i > 0) return 'há ' . $diff->i . ' min';
            return 'há ' . $diff->s . ' seg';
        } catch (Exception $e) { return 'N/A'; }
    }
    
    $activity_log = [];
    foreach ($conexoes_detalhadas as $conexao) {
        $activity_log[] = [
            'usuario' => htmlspecialchars($conexao['usuario']),
            'ip' => htmlspecialchars($conexao['ip']),
            'assistindo' => htmlspecialchars($conexao['stream_name']),
            'ultima_atividade' => formatar_tempo_passado_api($conexao['ultima_atividade']),
            'tipo' => htmlspecialchars($conexao['stream_type'])
        ];
    }

    $top_content_log = [];
    foreach ($conteudo_popular as $conteudo) {
        $top_content_log[] = [
            'nome' => htmlspecialchars($conteudo['name']),
            'icon' => htmlspecialchars($conteudo['stream_icon']),
            'espectadores' => $conteudo['total_espectadores']
        ];
    }
    
    // --- RESPOSTA FINAL EM JSON ---
    $response = [
        'online_count' => count($conexoes_detalhadas),
        'usuarios_unicos' => count(array_unique(array_column($conexoes_detalhadas, 'usuario'))),
        'activity' => $activity_log,
        'top_channels' => $top_content_log
    ];
    
    echo json_encode($response);

} catch (Exception $e) {
    http_response_code(500); // Código "Erro Interno do Servidor"
    echo json_encode(['error' => 'Erro no servidor ao processar sua solicitação.']);
    // Em produção, o ideal é registrar o erro em um arquivo de log
    // error_log("API Dashboard Error: " . $e->getMessage());
}

// Fecha a conexão com o banco de dados
$pdo = null;
?>