<?php
// Arquivo: /api/personalizar.php
// [CORRIGIDO] Alterado o sistema de segurança para usar TOKEN + SESSÃO (SESSION).

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// --- SEGURANÇA ---
// [CORRIGIDO] Usa o Token E a SESSÃO para verificar o nível.

// 1. Carrega a conexão com o banco
try {
    require_once 'controles/db.php';
} catch (Exception $e) {
    header('Content-Type: application/json');
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Erro crítico: Não foi possível carregar o arquivo do banco de dados.']);
    exit;
}

// 2. Conecta ao BD
$conexao = conectar_bd();
if (!$conexao) {
    header('Content-Type: application/json');
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Falha na conexão com o banco de dados.']);
    exit;
}

// 3. Valida o Token
$token_enviado = $_POST['token'] ?? '';

if (empty($token_enviado)) {
    header('Content-Type: application/json');
    http_response_code(403); // Forbidden
    echo json_encode(['status' => 'error', 'message' => 'Acesso negado: Token de autenticação não fornecido.']);
    exit;
}

try {
    // Esta é a query correta (igual à api_migracao.php, sem a coluna 'nivel')
    $sql_token = "SELECT id FROM admin WHERE token = :token";
    $stmt_token = $conexao->prepare($sql_token);
    $stmt_token->bindParam(':token', $token_enviado, PDO::PARAM_STR);
    $stmt_token->execute();
    $admin_logado = $stmt_token->fetch(PDO::FETCH_ASSOC);

    if (!$admin_logado || empty($admin_logado['id'])) {
        header('Content-Type: application/json');
        http_response_code(403); // Forbidden
        echo json_encode(['status' => 'error', 'message' => 'Acesso negado: Token inválido ou expirado. Faça login novamente.']);
        exit;
    }
    
    // [CORREÇÃO] VERIFICA O NÍVEL PELA SESSÃO (SESSION)
    // Esta é a forma correta que seu painel usa.
    if (!isset($_SESSION['nivel_admin']) || $_SESSION['nivel_admin'] != 1) {
         header('Content-Type: application/json');
        http_response_code(403); // Forbidden
        echo json_encode(['status' => 'error', 'message' => 'Acesso negado: Apenas o Administrador principal pode alterar estas configurações.']);
        exit;
    }

} catch (PDOException $e) {
    header('Content-Type: application/json');
    http_response_code(500);
    // Esta é a mensagem que você viu.
    echo json_encode(['status' => 'error', 'message' => 'Erro no banco de dados durante a validação do token. (' . $e->getMessage() . ')']);
    exit;
}
// --- FIM DA SEGURANÇA ---


// Define o cabeçalho como JSON para todas as respostas
header('Content-Type: application/json');

// --- FUNÇÃO DE RESPOSTA ---
function send_json_response($status, $message) {
    if ($status === 'error') {
        http_response_code(400); 
    }
    echo json_encode(['status' => $status, 'message' => $message]);
    exit();
}

// --- CAMINHOS ---
$config_file_path = __DIR__ . '/../config.json';
$upload_dir = __DIR__ . '/../img/'; 

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    send_json_response('error', 'Método de requisição inválido. Use POST.');
}

if (!isset($_POST['action'])) {
    send_json_response('error', 'Nenhuma ação especificada.');
}

// --- LÓGICA PRINCIPAL ---

if (!file_exists($config_file_path)) {
    http_response_code(500);
    send_json_response('error', 'Erro Crítico: Arquivo de configuração (config.json) não encontrado.');
}
if (!is_readable($config_file_path)) {
    http_response_code(500);
    send_json_response('error', 'Erro Crítico: O config.json não tem permissão de LEITURA. (CHMOD 644)');
}

$config_json = file_get_contents($config_file_path);
$config = json_decode($config_json, true);

if ($config === null) {
    http_response_code(500);
    send_json_response('error', 'Erro Crítico: Falha ao ler o arquivo de configuração (JSON inválido).');
}

$action = $_POST['action'];

switch ($action) {
    
    case 'update_title':
        if (!isset($_POST['titulo']) || empty(trim($_POST['titulo']))) {
            send_json_response('error', 'O campo título não pode estar vazio.');
        }
        
        if (!is_writable($config_file_path)) {
            http_response_code(500);
            send_json_response('error', 'Erro de Permissão: O arquivo config.json não tem permissão de ESCRITA. (CHMOD 664)');
        }

        $new_title = htmlspecialchars(trim($_POST['titulo']));
        $config['title'] = $new_title; 

        if (file_put_contents($config_file_path, json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
            send_json_response('success', 'Título do site atualizado com sucesso!');
        } else {
            http_response_code(500);
            send_json_response('error', 'Falha ao salvar o arquivo de configuração. Verifique as permissões do servidor.');
        }
        break;

    case 'update_logo':
        if (!isset($_FILES['logo']) || $_FILES['logo']['error'] !== UPLOAD_ERR_OK) {
            send_json_response('error', 'Falha no upload do logo. (Erro PHP: ' . $_FILES['logo']['error'] . ')');
        }

        if (!is_dir($upload_dir)) {
            if (!@mkdir($upload_dir, 0755, true)) {
                http_response_code(500);
                send_json_response('error', 'Erro de Permissão: Falha ao criar a pasta /img/. Verifique as permissões da pasta raiz.');
            }
        }
        if (!is_writable($upload_dir)) {
            http_response_code(500);
            send_json_response('error', 'Erro de Permissão: A pasta /img/ não tem permissão de escrita. (CHMOD 755)');
        }
         if (!is_writable($config_file_path)) {
            http_response_code(500);
            send_json_response('error', 'Erro de Permissão: O arquivo config.json não tem permissão de ESCRITA. (CHMOD 664)');
        }

        $file = $_FILES['logo'];
        
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'svg', 'webp'];
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

        if (function_exists('finfo_open')) {
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/svg+xml', 'image/webp'];
            $file_info = new finfo(FILEINFO_MIME_TYPE);
            $mime_type = $file_info->buffer(file_get_contents($file['tmp_name']));
            
            if (!in_array($mime_type, $allowed_types) || !in_array($extension, $allowed_extensions)) {
                 send_json_response('error', 'Tipo de arquivo inválido. Apenas JPG, PNG, GIF, SVG ou WEBP são permitidos.');
            }
        } else {
             if (!in_array($extension, $allowed_extensions)) {
                 send_json_response('error', 'Extensão de arquivo inválida. Apenas JPG, PNG, GIF, SVG ou WEBP são permitidos.');
            }
        }
        
        if ($file['size'] > 5 * 1024 * 1024) { // 5MB
             send_json_response('error', 'Arquivo muito grande. O limite é de 5MB.');
        }

        $new_filename = 'logo.' . $extension;
        $new_filepath_server = $upload_dir . $new_filename; 
        
        // [CORRIGIDO] Salva o caminho no mesmo formato do seu config.json original (./img/...)
        $new_filepath_web = './img/' . $new_filename; 

        if (move_uploaded_file($file['tmp_name'], $new_filepath_server)) {
            $config['logo_path'] = $new_filepath_web;

            if (file_put_contents($config_file_path, json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                send_json_response('success', 'Logo atualizado com sucesso!');
            } else {
                http_response_code(500);
                send_json_response('error', 'Logo enviado, mas falha ao salvar o config.json. Verifique as permissões.');
            }
        } else {
            http_response_code(500);
            send_json_response('error', 'Falha ao mover o arquivo enviado. Verifique as permissões (CHMOD) da pasta /img/.');
        }
        break;

    default:
        send_json_response('error', 'Ação desconhecida.');
        break;
}

?>