<?php
// Arquivo: /api/personalizar.php
// Este arquivo recebe as requisições AJAX da página de personalização.
// [CORRIGIDO] Adicionada verificação de 'fileinfo' e permissões de pasta.

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// --- SEGURANÇA ---
// Verifica se o usuário é um administrador logado (ESSENCIAL)
if (empty($_SESSION['admin_id'])) {
    // Define o cabeçalho como JSON e retorna um erro de não autorizado
    header('Content-Type: application/json');
    http_response_code(403); // Forbidden
    echo json_encode([
        'status' => 'error',
        'message' => 'Acesso negado. Sua sessão de administrador expirou ou é inválida.'
    ]);
    exit();
}

// Define o cabeçalho como JSON para todas as respostas
header('Content-Type: application/json');

// --- FUNÇÃO DE RESPOSTA ---
// Função auxiliar para enviar respostas JSON padronizadas e sair
function send_json_response($status, $message) {
    // Garante que o código HTTP correto seja enviado (útil para o JavaScript)
    if ($status === 'error') {
        http_response_code(400); // Bad Request (erro do usuário)
    }
    echo json_encode(['status' => $status, 'message' => $message]);
    exit();
}

// --- CAMINHOS ---
// O config.json está um nível ACIMA da pasta /api/
$config_file_path = __DIR__ . '/../config.json';
// O diretório de upload agora é a pasta /img/
$upload_dir = __DIR__ . '/../img/'; 

// Verifica se o método é POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    send_json_response('error', 'Método de requisição inválido. Use POST.');
}

// Verifica se a ação foi definida
if (!isset($_POST['action'])) {
    send_json_response('error', 'Nenhuma ação especificada.');
}

// --- LÓGICA PRINCIPAL ---

// Tenta carregar o config.json
if (!file_exists($config_file_path)) {
    http_response_code(500); // Erro do servidor
    send_json_response('error', 'Erro Crítico: Arquivo de configuração (config.json) não encontrado.');
}
if (!is_readable($config_file_path)) {
    http_response_code(500);
    send_json_response('error', 'Erro Crítico: O config.json não tem permissão de LEITURA. (CHMOD 644)');
}

$config_json = file_get_contents($config_file_path);
$config = json_decode($config_json, true);

if ($config === null) {
    http_response_code(500);
    send_json_response('error', 'Erro Crítico: Falha ao ler o arquivo de configuração (JSON inválido).');
}

$action = $_POST['action'];

switch ($action) {
    
    // --- CASO: ATUALIZAR TÍTULO ---
    case 'update_title':
        if (!isset($_POST['titulo']) || empty(trim($_POST['titulo']))) {
            send_json_response('error', 'O campo título não pode estar vazio.');
        }
        
        if (!is_writable($config_file_path)) {
            http_response_code(500);
            send_json_response('error', 'Erro de Permissão: O arquivo config.json não tem permissão de ESCRITA. (CHMOD 664)');
        }

        $new_title = htmlspecialchars(trim($_POST['titulo']));
        $config['title'] = $new_title; // Atualiza o valor no array PHP

        // Tenta salvar o arquivo JSON
        if (file_put_contents($config_file_path, json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
            send_json_response('success', 'Título do site atualizado com sucesso!');
        } else {
            http_response_code(500);
            send_json_response('error', 'Falha ao salvar o arquivo de configuração. Verifique as permissões do servidor.');
        }
        break;

    // --- CASO: ATUALIZAR LOGO ---
    case 'update_logo':
        if (!isset($_FILES['logo']) || $_FILES['logo']['error'] !== UPLOAD_ERR_OK) {
            send_json_response('error', 'Falha no upload do logo. (Erro PHP: ' . $_FILES['logo']['error'] . ')');
        }

        // [CORREÇÃO] Verifica se a pasta /img/ existe
        if (!is_dir($upload_dir)) {
            if (!@mkdir($upload_dir, 0755, true)) {
                http_response_code(500);
                send_json_response('error', 'Erro de Permissão: Falha ao criar a pasta /img/. Verifique as permissões da pasta raiz.');
            }
        }
        // [CORREÇÃO] Verifica se a pasta /img/ tem permissão de escrita
        if (!is_writable($upload_dir)) {
            http_response_code(500);
            send_json_response('error', 'Erro de Permissão: A pasta /img/ não tem permissão de escrita. (CHMOD 755)');
        }
        // [CORREÇÃO] Verifica se o config.json tem permissão de escrita
         if (!is_writable($config_file_path)) {
            http_response_code(500);
            send_json_response('error', 'Erro de Permissão: O arquivo config.json não tem permissão de ESCRITA. (CHMOD 664)');
        }

        $file = $_FILES['logo'];
        
        // Validação de segurança
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'svg', 'webp'];
        $extension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));

        // [CORREÇÃO] Adicionada verificação da extensão fileinfo (se existir)
        if (function_exists('finfo_open')) {
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif', 'image/svg+xml', 'image/webp'];
            $file_info = new finfo(FILEINFO_MIME_TYPE);
            $mime_type = $file_info->buffer(file_get_contents($file['tmp_name']));
            
            if (!in_array($mime_type, $allowed_types) || !in_array($extension, $allowed_extensions)) {
                 send_json_response('error', 'Tipo de arquivo inválido. Apenas JPG, PNG, GIF, SVG ou WEBP são permitidos.');
            }
        } else {
            // Fallback se a extensão fileinfo não estiver instalada
             if (!in_array($extension, $allowed_extensions)) {
                 send_json_response('error', 'Extensão de arquivo inválida. Apenas JPG, PNG, GIF, SVG ou WEBP são permitidos.');
            }
        }
        
        if ($file['size'] > 5 * 1024 * 1024) { // Limite de 5MB
             send_json_response('error', 'Arquivo muito grande. O limite é de 5MB.');
        }

        $new_filename = 'logo.' . $extension;
        $new_filepath_server = $upload_dir . $new_filename; 
        $new_filepath_web = '/img/' . $new_filename; // Caminho relativo à raiz

        // Tenta mover o arquivo
        if (move_uploaded_file($file['tmp_name'], $new_filepath_server)) {
            // Sucesso! Agora atualiza o config.json
            $config['logo_path'] = $new_filepath_web;

            if (file_put_contents($config_file_path, json_encode($config, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE))) {
                send_json_response('success', 'Logo atualizado com sucesso!');
            } else {
                http_response_code(500);
                send_json_response('error', 'Logo enviado, mas falha ao salvar o config.json. Verifique as permissões.');
            }
        } else {
            http_response_code(500);
            send_json_response('error', 'Falha ao mover o arquivo enviado. Verifique as permissões (CHMOD) da pasta /img/.');
        }
        break;

    // --- CASO PADRÃO ---
    default:
        send_json_response('error', 'Ação desconhecida.');
        break;
}

?>