<?php
/**
 * ARQUIVO: /gerenciador/ajax/series_excluir_massa.php
 * DESCRIÇÃO: Script para exclusão em massa de séries, incluindo suas temporadas e episódios.
 */

// Define o tipo de conteúdo da resposta como JSON para comunicação com o frontend.
header('Content-Type: application/json');

// Inclui o arquivo de conexão com o banco de dados e estabelece a conexão.
require_once('../../api/controles/db.php');
$pdo = conectar_bd();

// Verifica se a conexão com o banco de dados foi bem-sucedida.
if (!$pdo) {
    echo json_encode(['success' => false, 'message' => 'Erro ao conectar com o banco de dados.']);
    exit;
}

// Obtém os IDs enviados via POST.
$ids = $_POST['ids'] ?? null;

// Validação inicial: verifica se os IDs foram enviados e se é um array.
if (empty($ids) || !is_array($ids)) {
    echo json_encode(['success' => false, 'message' => 'Nenhum ID de série foi fornecido.']);
    exit;
}

// Filtra o array para garantir que todos os valores sejam numéricos, removendo quaisquer valores inválidos.
$ids = array_filter($ids, 'is_numeric');

// Validação final: verifica se restou algum ID válido após a filtragem.
if (empty($ids)) {
    echo json_encode(['success' => false, 'message' => 'Nenhum ID de série válido foi fornecido.']);
    exit;
}

try {
    // Cria uma string de placeholders (?) para usar na cláusula IN da query SQL.
    // Isso previne injeção de SQL e torna a query mais segura e eficiente.
    $placeholders = implode(',', array_fill(0, count($ids), '?'));
    
    // Inicia uma transação. Isso garante que todas as operações de exclusão
    // sejam executadas com sucesso. Se uma falhar, todas são revertidas (rollback).
    $pdo->beginTransaction();

    // Passo 1: Excluir os episódios relacionados às séries.
    // Esta é uma tabela "filha" e deve ser limpa antes da tabela "pai".
    $sql_episodes = "DELETE FROM series_episodes WHERE series_id IN ($placeholders)";
    $stmt_episodes = $pdo->prepare($sql_episodes);
    $stmt_episodes->execute($ids);

    // Passo 2: Excluir as temporadas relacionadas às séries.
    // Esta é outra tabela "filha".
    $sql_seasons = "DELETE FROM series_seasons WHERE series_id IN ($placeholders)";
    $stmt_seasons = $pdo->prepare($sql_seasons);
    $stmt_seasons->execute($ids);

    // Passo 3: Excluir as séries principais.
    // Esta é a tabela "pai" e deve ser a última a ter seus registros removidos.
    $sql_series = "DELETE FROM series WHERE id IN ($placeholders)";
    $stmt_series = $pdo->prepare($sql_series);
    $stmt_series->execute($ids);

    // Se todas as queries foram executadas sem erro, confirma as alterações no banco de dados.
    $pdo->commit();

    // Retorna uma mensagem de sucesso para o frontend, informando quantos itens foram excluídos.
    echo json_encode(['success' => true, 'message' => $stmt_series->rowCount() . ' série(s), suas temporadas e episódios foram excluído(s) com sucesso!']);

} catch (PDOException $e) {
    // Se ocorrer qualquer erro durante a execução das queries, reverte todas as alterações.
    $pdo->rollBack();
    
    // Retorna uma mensagem de erro detalhada para o frontend.
    // Em um ambiente de produção, pode ser útil logar o $e->getMessage() em vez de exibi-lo ao usuário.
    echo json_encode(['success' => false, 'message' => 'Erro no banco de dados: ' . $e->getMessage()]);
}
?>