<?php
/**
 * ARQUIVO: /gerenciador/ajax/temporadas_excluir_massa.php
 * DESCRIÇÃO: Script para exclusão em massa de temporadas e seus respectivos episódios.
 */

// Define o tipo de conteúdo da resposta como JSON.
header('Content-Type: application/json');

// Inclui o arquivo de conexão com o banco de dados.
require_once('../../api/controles/db.php');
$pdo = conectar_bd();

// Verifica a conexão com o banco de dados.
if (!$pdo) {
    echo json_encode(['success' => false, 'message' => 'Erro ao conectar com o banco de dados.']);
    exit;
}

// Obtém os IDs das temporadas enviados via POST.
$ids = $_POST['ids'] ?? null;

// Validação: verifica se os IDs foram enviados.
if (empty($ids) || !is_array($ids)) {
    echo json_encode(['success' => false, 'message' => 'Nenhum ID de temporada foi fornecido.']);
    exit;
}

// Garante que todos os valores sejam numéricos.
$ids = array_filter($ids, 'is_numeric');

// Validação final: verifica se restou algum ID válido.
if (empty($ids)) {
    echo json_encode(['success' => false, 'message' => 'Nenhum ID de temporada válido foi fornecido.']);
    exit;
}

try {
    // Cria placeholders (?) para a cláusula IN, evitando injeção de SQL.
    $placeholders = implode(',', array_fill(0, count($ids), '?'));
    
    // Inicia uma transação para garantir a integridade dos dados.
    $pdo->beginTransaction();

    // Passo 1: Excluir os episódios relacionados às temporadas.
    // Presumindo que sua tabela de episódios tenha uma coluna `season_id` para relacionar com a temporada.
    // Se o nome da coluna for diferente, ajuste a query.
    $sql_episodes = "DELETE FROM series_episodes WHERE season_id IN ($placeholders)";
    $stmt_episodes = $pdo->prepare($sql_episodes);
    $stmt_episodes->execute($ids);

    // Passo 2: Excluir as temporadas principais.
    $sql_seasons = "DELETE FROM series_seasons WHERE id IN ($placeholders)";
    $stmt_seasons = $pdo->prepare($sql_seasons);
    $stmt_seasons->execute($ids);

    // Confirma as alterações no banco de dados.
    $pdo->commit();

    // Retorna uma mensagem de sucesso.
    echo json_encode(['success' => true, 'message' => $stmt_seasons->rowCount() . ' temporada(s) e seus episódios foram excluído(s) com sucesso!']);

} catch (PDOException $e) {
    // Se ocorrer um erro, reverte todas as alterações.
    $pdo->rollBack();
    
    // Retorna uma mensagem de erro.
    echo json_encode(['success' => false, 'message' => 'Erro no banco de dados: ' . $e->getMessage()]);
}
?>