<?php
// ARQUIVO: /gerenciador/episodios_gerenciar.php
require_once('../api/controles/db.php');
$pdo = conectar_bd();

// Pega o ID da temporada da URL. Este é o ID único da tabela 'series_seasons'.
$season_id = filter_input(INPUT_GET, 'season_id', FILTER_VALIDATE_INT);
if (!$season_id) {
    die("ID da temporada inválido ou não fornecido.");
}

// Busca os detalhes da temporada (como o nome e número) e o nome da série a que ela pertence.
$stmt_season = $pdo->prepare(
    "SELECT ss.id, ss.series_id, ss.season_number, ss.name, s.name as series_name
     FROM series_seasons ss
     JOIN series s ON ss.series_id = s.id
     WHERE ss.id = ?"
);
$stmt_season->execute([$season_id]);
$temporada = $stmt_season->fetch(PDO::FETCH_ASSOC);

if (!$temporada) {
    die("Temporada não encontrada.");
}

// CORREÇÃO PRINCIPAL APLICADA AQUI
// Buscamos os episódios usando o ID da SÉRIE e o NÚMERO da temporada,
// que é a forma correta de relacionar os dados.
$stmt_episodes = $pdo->prepare(
    "SELECT * FROM series_episodes
     WHERE series_id = ? AND season = ?
     ORDER BY episode_num ASC"
);
$stmt_episodes->execute([$temporada['series_id'], $temporada['season_number']]);
$episodios = $stmt_episodes->fetchAll(PDO::FETCH_ASSOC);

?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Episódios</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
    <style>
        body { background-color: #f0f2f5; }
        .card { border: none; border-radius: 0.75rem; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
        .action-btn { width: 38px; height: 38px; border-radius: 50%; display: inline-flex; align-items: center; justify-content: center; margin: 0 3px; border: none; color: white; text-decoration: none; transition: transform 0.2s; cursor: pointer; }
        .action-btn:hover { transform: scale(1.1); }
        .btn-play { background-color: #198754; } .btn-edit { background-color: #0d6efd; } .btn-delete { background-color: #dc3545; }
        
        /* Estilos do tema escuro */
        body.dark-mode { background-color: #121212; color: #e0e0e0; }
        body.dark-mode .card { background-color: #1e1e1e; border: 1px solid #2c2c2c; }
        body.dark-mode .card-header, body.dark-mode h4, body.dark-mode .btn-outline-secondary { color: #ffffff; border-color: #444; }
        body.dark-mode .table { color: #e0e0e0; }
        body.dark-mode .table-light { --bs-table-bg: #2a2a2a; --bs-table-border-color: #3c3c3c; color: #ffffff; }
        body.dark-mode .table-hover tbody tr:hover { background-color: #2c2c2c; }
        body.dark-mode .table .table-light th { color: #f8f9fa; font-weight: 600; }
    </style>
</head>
<body>
<div class="container-fluid p-md-4 p-2">
    <div class="card">
        <div class="card-header bg-transparent p-3 d-flex justify-content-between align-items-center">
            <div>
                <h4 class="mb-0"><i class="fas fa-film me-2"></i>Episódios de: <strong><?= htmlspecialchars($temporada['series_name']) ?> - <?= htmlspecialchars($temporada['name']) ?></strong></h4>
                <a href="serie_episodios.php?id=<?= htmlspecialchars($temporada['series_id']) ?>" class="btn btn-sm btn-outline-secondary mt-2"><i class="fas fa-arrow-left me-2"></i>Voltar para Temporadas</a>
            </div>
            
            <div>
                <button onclick="history.back()" class="btn btn-info"><i class="fas fa-reply me-2"></i>Voltar</button>
                
                <a href="episodio_adicionar.php?season_id=<?= htmlspecialchars($season_id) ?>" class="btn btn-primary"><i class="fas fa-plus me-2"></i>Novo Episódio</a>
            </div>
        </div>
        <div class="card-body">
            <div class="mb-3">
                <button id="bulkDeleteButton" class="btn btn-danger" style="display: none;"><i class="fas fa-trash-alt me-2"></i>Excluir Selecionados</button>
            </div>
            <div class="table-responsive">
                <table class="table table-hover align-middle">
                    <thead class="table-light">
                        <tr>
                            <th style="width: 5%;"><input type="checkbox" class="form-check-input" id="selectAllCheckbox"></th>
                            <th style="width: 5%;">#</th>
                            <th>Título</th>
                            <th style="width: 15%;">Duração</th>
                            <th style="width: 20%;" class="text-center">Ações</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php if (empty($episodios)): ?>
                            <tr><td colspan="5" class="text-center py-4">Nenhum episódio encontrado.</td></tr>
                        <?php else: foreach ($episodios as $ep): ?>
                        <tr id="episode-row-<?= htmlspecialchars($ep['id']) ?>">
                            <td><input type="checkbox" class="form-check-input row-checkbox" data-id="<?= htmlspecialchars($ep['id']) ?>"></td>
                            <td><?= htmlspecialchars($ep['episode_num']) ?></td>
                            <td><?= htmlspecialchars($ep['title']) ?></td>
                            <td><?= htmlspecialchars(gmdate("H:i:s", $ep['duration_secs'] ?? 0)) ?></td>
                            <td class="text-center">
                                <a href="<?= htmlspecialchars($ep['link']) ?>" target="_blank" class="action-btn btn-play" title="Testar Link"><i class="fas fa-play"></i></a>
                                <a href="episodio_editar.php?id=<?= htmlspecialchars($ep['id']) ?>" class="action-btn btn-edit" title="Editar Episódio"><i class="fas fa-pencil-alt"></i></a>
                                <button onclick="deleteEpisodio(<?= htmlspecialchars($ep['id']) ?>)" class="action-btn btn-delete" title="Excluir Episódio"><i class="fas fa-trash"></i></button>
                            </td>
                        </tr>
                        <?php endforeach; endif; ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
// --- JAVASCRIPT DO TEMA ---
(function() { const themeKey = 'theme'; function applyTheme() { const savedTheme = localStorage.getItem(themeKey); document.body.classList.toggle('dark-mode', savedTheme === 'dark'); } applyTheme(); })();

// --- LÓGICA PARA AÇÕES EM MASSA ---
// (Seu código JavaScript para exclusão em massa continua aqui, pois já está correto)
const selectAllCheckbox = document.getElementById('selectAllCheckbox');
const bulkDeleteButton = document.getElementById('bulkDeleteButton');
const rowCheckboxes = document.querySelectorAll('.row-checkbox');

function updateBulkActions() {
    const checkedCount = document.querySelectorAll('.row-checkbox:checked').length;
    bulkDeleteButton.style.display = checkedCount > 0 ? 'inline-block' : 'none';
    bulkDeleteButton.innerText = `Excluir Selecionados (${checkedCount})`;
}

selectAllCheckbox.addEventListener('change', function() {
    rowCheckboxes.forEach(cb => { cb.checked = this.checked; });
    updateBulkActions();
});

rowCheckboxes.forEach(cb => {
    cb.addEventListener('change', function() {
        if (!this.checked) {
            selectAllCheckbox.checked = false;
        } else {
            const allChecked = Array.from(rowCheckboxes).every(c => c.checked);
            selectAllCheckbox.checked = allChecked;
        }
        updateBulkActions();
    });
});

bulkDeleteButton.addEventListener('click', async function() {
    const checkedIds = Array.from(document.querySelectorAll('.row-checkbox:checked')).map(cb => cb.dataset.id);
    if (checkedIds.length === 0 || !confirm(`Tem certeza que deseja excluir ${checkedIds.length} episódio(s)?`)) return;

    try {
        const formData = new FormData();
        checkedIds.forEach(id => formData.append('ids[]', id));
        
        const response = await fetch('ajax/episodio_excluir_massa.php', { method: 'POST', body: formData });
        const result = await response.json();
        
        alert(result.message);
        if (result.success) location.reload();
    } catch (error) {
        alert('Erro de comunicação ao tentar excluir os episódios.');
    }
});

// --- FUNÇÃO PARA EXCLUIR UM ÚNICO EPISÓDIO ---
async function deleteEpisodio(id) {
    if (!confirm(`Tem certeza que deseja excluir este episódio?`)) return;
    try {
        const formData = new FormData();
        formData.append('id', id);

        const response = await fetch('ajax/episodio_excluir.php', { method: 'POST', body: formData });
        const result = await response.json();
        
        alert(result.message);
        if (result.success) {
            document.getElementById(`episode-row-${id}`).remove();
            updateBulkActions();
        }
    } catch (e) {
        alert('Erro de comunicação com o servidor.');
    }
}
updateBulkActions();
</script>
</body>
</html>