<?php
// ARQUIVO: /gerenciador/filmes.php
ini_set('display_errors', 1); 
error_reporting(E_ALL);

// --- VERIFICAÇÃO E CONEXÃO SEGURA ---
$pdo = null;
$error_message = '';
$filmes = [];
$categorias = [];
$total_pages = 0;
$page = 1;
$limit = 50;
$searchTermo = '';
$searchCategoria = '';
$allowed_limits = [10, 50, 250, 500, 1000, 2000, 3000];

try {
    $db_path = '../api/controles/db.php';

    if (!file_exists($db_path)) {
        throw new Exception("Erro Crítico: O arquivo de conexão não foi encontrado no caminho esperado: '{$db_path}'.");
    }
    require_once($db_path);

    if (!function_exists('conectar_bd')) {
        throw new Exception("Erro Crítico: Função conectar_bd() não encontrada.");
    }

    $pdo = conectar_bd();
    if (!$pdo) {
        throw new Exception("Falha na conexão com o banco de dados.");
    }

    // --- PAGINAÇÃO E FILTROS ---
    $limit = (isset($_GET['limit']) && in_array($_GET['limit'], $allowed_limits)) ? (int)$_GET['limit'] : 50;
    $page = isset($_GET['page']) ? max(1, (int)$_GET['page']) : 1;
    $offset = ($page - 1) * $limit;

    $searchTermo = $_GET['busca'] ?? '';
    $searchCategoria = $_GET['categoria_id'] ?? '';
    $params = [];

    $base_where = " WHERE f.stream_type = 'movie' ";
    $whereClauses = [];

    if (!empty($searchTermo)) {
        $whereClauses[] = "f.name LIKE :termo";
        $params[':termo'] = '%' . $searchTermo . '%';
    }
    if (!empty($searchCategoria)) {
        $whereClauses[] = "f.category_id = :categoria_id";
        $params[':categoria_id'] = $searchCategoria;
    }

    $where_string = empty($whereClauses) ? '' : ' AND ' . implode(' AND ', $whereClauses);

    $count_sql = "SELECT COUNT(f.id) FROM streams f" . $base_where . $where_string;
    $count_stmt = $pdo->prepare($count_sql);
    $count_stmt->execute($params);
    $total_filmes = $count_stmt->fetchColumn();
    $total_pages = ceil($total_filmes / $limit);

    $sql = "SELECT f.id, f.name, f.stream_icon, f.link, f.year, c.nome AS category_name 
            FROM streams f 
            LEFT JOIN categoria c ON f.category_id = c.id" . $base_where . $where_string . " ORDER BY f.id DESC LIMIT :limit OFFSET :offset";

    $stmt = $pdo->prepare($sql);
    if (!empty($params)) { foreach ($params as $key => $val) { $stmt->bindValue($key, $val); } }
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT); 
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    $filmes = $stmt->fetchAll(PDO::FETCH_ASSOC);

    $categorias = $pdo->query("SELECT id, nome FROM categoria WHERE type = 'movie' ORDER BY nome")->fetchAll(PDO::FETCH_ASSOC);

} catch (Exception $e) {
    $error_message = $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Gerenciar Filmes</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.2/css/all.min.css">
    
    <style>
        /* Estilos Gerais */
        body { background-color: #f0f2f5; }
        .card { border: none; border-radius: 0.75rem; box-shadow: 0 4px 6px rgba(0,0,0,0.1); }
        .table-hover tbody tr:hover { background-color: #f8f9fa; }
        
        /* BOTÕES DE AÇÃO - CSS CRÍTICO */
        .action-btn { 
            width: 38px; 
            height: 38px; 
            border-radius: 50%; 
            display: inline-flex; 
            align-items: center; 
            justify-content: center; 
            margin: 0 3px; 
            border: none; 
            color: white !important; 
            text-decoration: none; 
            transition: transform 0.2s; 
            cursor: pointer; 
        }
        .action-btn:hover { transform: scale(1.1); color: white; }
        
        .btn-edit { background-color: #0d6efd; } /* Azul */
        .btn-delete { background-color: #dc3545; } /* Vermelho */
        
        .badge.bg-primary { background-color: #6f42c1 !important; }
        .poster-img { height: 50px; width: auto; max-width: 40px; object-fit: cover; border-radius: 5px; background-color: #e9ecef; vertical-align: middle; }
        
        @media (max-width: 768px) { 
            .mobile-hide { display: none; } 
            .action-btn { width: 34px; height: 34px; margin: 0 2px; } 
        }
        
        /* Dark Mode */
        body.dark-mode { background-color: #121212; color: #e0e0e0; }
        body.dark-mode .card { background-color: #1e1e1e; border: 1px solid #2c2c2c; }
        body.dark-mode .card-header { background-color: #1e1e1e !important; border-bottom: 1px solid #2c2c2c; }
        body.dark-mode .card-header h4 { color: #ffffff; }
        body.dark-mode .table { border-color: #3c3c3c; } 
        body.dark-mode thead.table-light th { background-color: #2a2a2a; color: #ffffff; border-color: #3c3c3c; }
        body.dark-mode .table tbody td { background-color: transparent; border-color: #3c3c3c; color: #f5f5f5; }
        body.dark-mode .table-hover tbody tr:hover { background-color: #2c2c2c; }
        body.dark-mode .form-control, body.dark-mode .form-select { background-color: #2a2a2a; color: #f5f5f5; border-color: #3c3c3c; }
        body.dark-mode .form-control::placeholder { color: #999; opacity: 1; }
        body.dark-mode .pagination .page-link { background-color: #2a2a2a; border-color: #3c3c3c; }
        body.dark-mode .pagination .page-item.active .page-link { background-color: #0d6efd; border-color: #0d6efd; }
        body.dark-mode .pagination .page-item.disabled .page-link { background-color: #1e1e1e; border-color: #3c3c3c; }
        body.dark-mode #theme-toggle-btn { background-color: #2a2a2a; color: #e0e0e0; border-color: #3c3c3c; }
    </style>
</head>
<body>
    <div class="container-fluid p-md-4 p-2">
        <?php if (!empty($error_message)): ?>
            <div class="alert alert-danger">
                <h4>Erro</h4>
                <pre><?= htmlspecialchars($error_message) ?></pre>
            </div>
        <?php else: ?>
            <div class="card">
                <div class="card-header p-3 d-flex justify-content-between align-items-center">
                    <h4 class="mb-0"><i class="fas fa-film me-2"></i>Gerenciar Filmes</h4>
                    <div>
                        <button id="theme-toggle-btn" class="btn btn-outline-secondary me-2"><i class="fas fa-moon"></i></button>
                        <a href="../dashboard.php" class="btn btn-secondary"><i class="fas fa-tachometer-alt me-2"></i>Voltar</a>
                        <a href="filme_novo.php" class="btn btn-primary"><i class="fas fa-plus me-2"></i>Novo</a>
                    </div>
                </div>
                <div class="card-body">
                    <form method="GET" action="filmes.php" class="row g-3 mb-4 align-items-center">
                        <div class="col-lg-3 col-md-6 mb-2"><select name="categoria_id" class="form-select" onchange="this.form.submit()"><option value="">Todas as categorias</option><?php foreach ($categorias as $cat): ?><option value="<?= htmlspecialchars($cat['id']) ?>" <?= ($searchCategoria == $cat['id']) ? 'selected' : '' ?>><?= htmlspecialchars($cat['nome']) ?></option><?php endforeach; ?></select></div>
                        <div class="col-lg-4 col-md-6 mb-2"><input type="text" name="busca" class="form-control" placeholder="Pesquisar filme..." value="<?= htmlspecialchars($searchTermo) ?>"></div>
                        <div class="col-lg-2 col-md-6 mb-2"><select name="limit" class="form-select" onchange="this.form.submit()"><?php foreach($allowed_limits as $lim): ?><option value="<?= $lim ?>" <?= ($limit == $lim) ? 'selected' : '' ?>><?= $lim ?> por página</option><?php endforeach; ?></select></div>
                        <div class="col-lg-3 col-md-6 mb-2"><button type="submit" class="btn btn-primary w-100">Buscar</button></div>
                    </form>
                    
                    <div class="mb-3">
                        <button id="bulkDeleteButton" class="btn btn-danger" style="display: none;"><i class="fas fa-trash-alt me-2"></i>Excluir Selecionados</button>
                        <button id="bulkMoveButton" class="btn btn-warning" style="display: none;"><i class="fas fa-random me-2"></i>Mover Selecionados</button>
                    </div>

                    <div class="table-responsive">
                        <table class="table table-hover align-middle">
                            <thead class="table-light">
                                <tr>
                                    <th><input type="checkbox" class="form-check-input" id="selectAllCheckbox"></th>
                                    <th class="mobile-hide">ID</th><th>Título</th><th>Poster</th><th class="mobile-hide">Categoria</th><th class="mobile-hide">Ano</th><th class="text-center">Ações</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($filmes)): ?><tr><td colspan="7" class="text-center">Nenhum filme encontrado.</td></tr>
                                <?php else: foreach ($filmes as $filme): ?>
                                <tr id="filme-row-<?= htmlspecialchars($filme['id']) ?>">
                                    <td><input type="checkbox" class="form-check-input row-checkbox" data-id="<?= htmlspecialchars($filme['id']) ?>"></td>
                                    <td class="mobile-hide"><?= htmlspecialchars($filme['id']) ?></td>
                                    <td><?= htmlspecialchars($filme['name']) ?></td>
                                    <td><img src="<?= htmlspecialchars($filme['stream_icon']) ?>" class="poster-img" alt="Poster"></td>
                                    <td class="mobile-hide"><span class="badge bg-primary"><?= htmlspecialchars($filme['category_name'] ?? 'N/A') ?></span></td>
                                    <td class="mobile-hide"><?= htmlspecialchars($filme['year'] ?? 'N/A') ?></td>
                                    <td class="text-center">
                                        
                                        <a href="filme_editar.php?id=<?= htmlspecialchars($filme['id']) ?>" class="action-btn btn-edit" title="Editar">
                                            <i class="fas fa-edit"></i>
                                        </a>
                                        
                                        <button onclick="deleteMovie(<?= htmlspecialchars($filme['id']) ?>, '<?= htmlspecialchars(addslashes($filme['name'])) ?>')" class="action-btn btn-delete" title="Excluir">
                                            <i class="fas fa-trash"></i>
                                        </button>

                                    </td>
                                </tr>
                                <?php endforeach; endif; ?>
                            </tbody>
                        </table>
                    </div>
                    <nav class="mt-3"><ul class="pagination justify-content-center"><?php if($page > 1): ?><li class="page-item"><a class="page-link" href="?page=<?= $page - 1 ?>&limit=<?= $limit ?>&busca=<?= $searchTermo ?>&categoria_id=<?= $searchCategoria ?>">Anterior</a></li><?php endif; ?><?php for($i = 1; $i <= $total_pages; $i++): ?><li class="page-item <?= ($i == $page) ? 'active' : '' ?>"><a class="page-link" href="?page=<?= $i ?>&limit=<?= $limit ?>&busca=<?= $searchTermo ?>&categoria_id=<?= $searchCategoria ?>"><?= $i ?></a></li><?php endfor; ?><?php if($page < $total_pages): ?><li class="page-item"><a class="page-link" href="?page=<?= $page + 1 ?>&limit=<?= $limit ?>&busca=<?= $searchTermo ?>&categoria_id=<?= $searchCategoria ?>">Próxima</a></li><?php endif; ?></ul></nav>
                </div>
            </div>
        <?php endif; ?>
    </div>

    <div class="modal fade" id="moveMoviesModal" tabindex="-1" aria-hidden="true">
      <div class="modal-dialog">
        <div class="modal-content">
          <div class="modal-header"><h5 class="modal-title">Mover Filmes</h5><button type="button" class="btn-close" data-bs-dismiss="modal"></button></div>
          <div class="modal-body">
            <p>Selecionados: <strong id="movieCount">0</strong></p>
            <select id="categorySelect" class="form-select">
                <option value="">-- Selecione a categoria --</option>
                <?php foreach ($categorias as $cat): ?><option value="<?= htmlspecialchars($cat['id']) ?>"><?= htmlspecialchars($cat['nome']) ?></option><?php endforeach; ?>
            </select>
          </div>
          <div class="modal-footer">
            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
            <button type="button" id="confirmMoveBtn" class="btn btn-primary">Mover</button>
          </div>
        </div>
      </div>
    </div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script>
    (function() {
        const themeKey = 'theme';
        function applyTheme() { document.body.classList.toggle('dark-mode', localStorage.getItem(themeKey) === 'dark'); }
        applyTheme();
        document.addEventListener('DOMContentLoaded', () => {
            const btn = document.getElementById('theme-toggle-btn');
            if(btn) btn.addEventListener('click', () => {
                const isDark = document.body.classList.contains('dark-mode');
                localStorage.setItem(themeKey, isDark ? 'light' : 'dark');
                applyTheme();
            });
        });
    })();

    document.addEventListener('DOMContentLoaded', function() {
        const selectAll = document.getElementById('selectAllCheckbox');
        const checkboxes = document.querySelectorAll('.row-checkbox');
        const btnDelete = document.getElementById('bulkDeleteButton');
        const btnMove = document.getElementById('bulkMoveButton');
        const moveModal = new bootstrap.Modal(document.getElementById('moveMoviesModal'));
        
        function updateButtons() {
            const count = document.querySelectorAll('.row-checkbox:checked').length;
            btnDelete.style.display = count > 0 ? 'inline-block' : 'none';
            btnMove.style.display = count > 0 ? 'inline-block' : 'none';
            document.getElementById('movieCount').textContent = count;
        }

        if (selectAll) {
            selectAll.addEventListener('change', function() {
                checkboxes.forEach(cb => cb.checked = this.checked);
                updateButtons();
            });
        }

        checkboxes.forEach(cb => cb.addEventListener('change', () => {
            updateButtons();
            if (selectAll) selectAll.checked = document.querySelectorAll('.row-checkbox:checked').length === checkboxes.length;
        }));

        function getIds() {
            return Array.from(document.querySelectorAll('.row-checkbox:checked')).map(cb => cb.getAttribute('data-id'));
        }

        if (btnDelete) {
            btnDelete.addEventListener('click', function() {
                const ids = getIds();
                if (ids.length === 0) return;

                if (confirm(`Excluir ${ids.length} filmes selecionados?`)) {
                    this.disabled = true; this.innerHTML = 'Excluindo...';
                    
                    fetch('/api/acao_filmes.php', { 
                        method: 'POST',
                        headers: { 'Content-Type': 'application/json' },
                        body: JSON.stringify({ acao: 'excluir', ids: ids })
                    })
                    .then(r => r.json())
                    .then(data => {
                        if(data.success) { alert(data.message); location.reload(); }
                        else { alert('Erro: ' + data.error); }
                    })
                    .catch(err => alert('Erro de conexão.'))
                    .finally(() => { this.disabled = false; this.innerHTML = 'Excluir Selecionados'; });
                }
            });
        }

        if (btnMove) {
            btnMove.addEventListener('click', () => moveModal.show());
            document.getElementById('confirmMoveBtn').addEventListener('click', function() {
                const ids = getIds();
                const catId = document.getElementById('categorySelect').value;
                if (!catId) return alert('Selecione a categoria.');

                this.disabled = true; this.textContent = 'Movendo...';
                
                fetch('/api/acao_filmes.php', { 
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ acao: 'mover', ids: ids, category_id: catId })
                })
                .then(r => r.json())
                .then(data => {
                    if(data.success) { alert(data.message); location.reload(); }
                    else { alert('Erro: ' + data.error); }
                })
                .catch(err => alert('Erro de conexão.'))
                .finally(() => { this.disabled = false; this.textContent = 'Mover'; moveModal.hide(); });
            });
        }
    });
    
    window.deleteMovie = function(id, name) {
        if (confirm(`Excluir "${name}"?`)) {
            fetch('/api/acao_filmes.php', { 
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ acao: 'excluir', ids: [id] })
            })
            .then(r => r.json())
            .then(data => {
                if(data.success) document.getElementById('filme-row-'+id).remove();
                else alert('Erro: ' + data.error);
            })
            .catch(err => alert('Erro de conexão.'));
        }
    };
</script>
</body>
</html>