<?php
session_start();
// Apenas Admins podem aceder a esta página
if (!isset($_SESSION['nivel_admin']) || $_SESSION['nivel_admin'] != 1) {
    header("Location: dashboard.php"); // Redireciona se não for admin
    exit;
}
require_once("menu.php");
?>

<style>
    /* ==============================================
       CORREÇÃO DE TEMA (CLARO/ESCURO)
       ============================================== */

    /* 1. Define as cores padrão (TEMA CLARO) */
    .ip-manager-container { 
        padding: 20px; 
        color: #212529; /* Cor do texto da página (escuro) */
    }
    .ip-table-wrapper {
        background-color: #fff; /* Cor do card (claro) */
        padding: 20px;
        border-radius: 8px;
        margin-bottom: 30px;
        border: 1px solid #dee2e6; /* Cor da borda (claro) */
        color: #212529; /* Cor do texto DENTRO do card (escuro) */
    }
    .ip-table th { 
        font-weight: 600; 
        color: #6c757d; /* Cor do cabeçalho da tabela (cinza escuro) */
    }
    .ip-table td {
        color: #212529; /* Cor do texto da célula (escuro) */
    }

    /* 2. Sobrescreve as cores para o TEMA ESCURO */
    [data-theme="dark"] .ip-manager-container {
        color: #e4e6eb; /* Cor do texto da página (claro) */
    }
    [data-theme="dark"] .ip-table-wrapper {
        background-color: #2a2e33; /* Cor do card (escuro) */
        border-color: #3a3f44; /* Cor da borda (escuro) */
        color: #e4e6eb; /* Cor do texto DENTRO do card (claro) */
    }
    [data-theme="dark"] .ip-table th {
        color: #b0b3b8; /* Cor do cabeçalho da tabela (cinza claro) */
    }
    [data-theme="dark"] .ip-table td {
        color: #e4e6eb; /* Cor do texto da célula (claro) */
    }

    /* ==============================================
       Estilos de Layout (Sem alteração)
       ============================================== */
    .ip-table { width: 100%; border-collapse: collapse; }
    .ip-table th, .ip-table td { 
        padding: 12px; 
        border-bottom: 1px solid var(--border-color, #dee2e6); 
        text-align: left; 
    }
    .whitelist-form, .manual-block-form { 
        display: flex; 
        gap: 10px; 
        margin-top: 10px; 
    }
</style>

<div class="ip-manager-container">
    <h2><i class="fas fa-shield-virus"></i> Gestor de IPs</h2>
    <p>Gira os IPs bloqueados automaticamente pelo sistema e a sua lista de permissões (whitelist).</p>

    <div class="ip-table-wrapper">
        <h3><i class="fas fa-user-lock"></i> IPs Atualmente Bloqueados</h3>
        <table class="ip-table">
            <thead>
                <tr>
                    <th><input type="checkbox" id="check-all-banned"></th>
                    <th>Endereço IP</th>
                    <th>Motivo</th>
                    <th>Data do Bloqueio</th>
                </tr>
            </thead>
            <tbody id="banned-ips-table"></tbody>
        </table>
        <button id="unblock-selected-btn" class="btn btn-warning mt-3">Desbloquear Selecionados</button>
    </div>

    <div class="ip-table-wrapper">
        <h3><i class="fas fa-ban text-danger"></i> Bloquear IP Manualmente</h3>
        <p>IPs bloqueados por aqui não são removidos automaticamente.</p>
        <div class="manual-block-form">
            <input type="text" id="manual-block-ip-input" class="form-control" placeholder="Bloquear IP (ex: 1.2.3.4)">
            <input type="text" id="manual-block-reason-input" class="form-control" placeholder="Motivo (ex: Admin Block)">
            <button id="add-manual-block-btn" class="btn btn-danger">Bloquear</button>
        </div>
    </div>

    <div class="ip-table-wrapper">
        <h3><i class="fas fa-shield-alt"></i> IPs Permitidos (Whitelist)</h3>
        <p>IPs nesta lista nunca serão bloqueados pelo sistema automático.</p>
        <div class="whitelist-form">
            <input type="text" id="whitelist-ip-input" class="form-control" placeholder="Adicionar IP (ex: 192.168.1.1)">
            <input type="text" id="whitelist-notes-input" class="form-control" placeholder="Notas (opcional)">
            <button id="add-whitelist-btn" class="btn btn-success">Adicionar</button>
        </div>
        <table class="ip-table mt-3">
            <thead>
                <tr>
                    <th>Endereço IP</th>
                    <th>Notas</th>
                    <th>Ação</th>
                </tr>
            </thead>
            <tbody id="allowed-ips-table"></tbody>
        </table>
    </div>
</div>

<?php
// Inclui o rodapé para carregar o SweetAlert (Swal)
require_once 'footer.php';
?>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const api = 'api/ip_manager.php';

    async function fetchData(action, body = {}) {
        // Adiciona o Token de sessão em todas as requisições
        body.token = (typeof SESSION_TOKEN !== 'undefined' ? SESSION_TOKEN : '');

        try {
            const response = await fetch(`${api}?action=${action}`, {
                method: 'POST',
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify(body)
            });
            
            if (!response.ok) {
                 const errorData = await response.json();
                 throw new Error(errorData.message || `Erro HTTP: ${response.status}`);
            }
            
            return response.json();

        } catch (error) {
            console.error('Erro no fetchData:', error);
            Swal.fire('Erro de Rede!', error.message, 'error');
            return null; // Retorna nulo para evitar que o código seguinte quebre
        }
    }

    function renderTables(data) {
        if (!data) return; // Não faz nada se a 'data' for nula (devido a um erro de rede)

        const bannedTable = document.getElementById('banned-ips-table');
        const allowedTable = document.getElementById('allowed-ips-table');
        
        bannedTable.innerHTML = '';
        if (data.banned && data.banned.length > 0) {
            data.banned.forEach(item => {
                bannedTable.innerHTML += `<tr>
                    <td><input type="checkbox" class="banned-ip-check" value="${item.ip_address}"></td>
                    <td>${item.ip_address}</td>
                    <td>${item.reason}</td>
                    <td>${item.ban_timestamp}</td>
                </tr>`;
            });
        } else {
            bannedTable.innerHTML = '<tr><td colspan="4">Nenhum IP bloqueado.</td></tr>';
        }

        allowedTable.innerHTML = '';
        if (data.allowed && data.allowed.length > 0) {
            data.allowed.forEach(item => {
                allowedTable.innerHTML += `<tr>
                    <td>${item.ip_address}</td>
                    <td>${item.notes}</td>
                    <td><button class="btn btn-danger btn-sm remove-allowed-btn" data-ip="${item.ip_address}">Remover</button></td>
                </tr>`;
            });
        } else {
            allowedTable.innerHTML = '<tr><td colspan="3">Nenhum IP na whitelist.</td></tr>';
        }
    }

    async function loadLists() {
        const data = await fetchData('get_lists');
        renderTables(data);
    }

    // Botão Desbloquear
    document.getElementById('unblock-selected-btn').addEventListener('click', async () => {
        const selectedIps = [...document.querySelectorAll('.banned-ip-check:checked')].map(cb => cb.value);
        if (selectedIps.length > 0) {
            const result = await fetchData('unblock_ips', { ips: selectedIps });
            if(result) {
                Swal.fire(result.status === 'success' ? 'Sucesso!' : 'Erro!', result.message || 'Operação concluída.', result.status || 'info');
                loadLists();
            }
        } else {
            Swal.fire('Atenção!', 'Por favor, selecione pelo menos um IP para desbloquear.', 'warning');
        }
    });

    // Botão Bloquear Manualmente
    document.getElementById('add-manual-block-btn').addEventListener('click', async () => {
        const ip = document.getElementById('manual-block-ip-input').value.trim();
        const reason = document.getElementById('manual-block-reason-input').value.trim() || 'Bloqueio Manual (Admin)'; // Motivo padrão
        
        if (ip) {
            const result = await fetchData('manual_block_ip', { ip, reason });
            if(result) {
                Swal.fire(result.status === 'success' ? 'Sucesso!' : 'Erro!', result.message || 'Operação concluída.', result.status || 'info');
                loadLists(); // Atualiza a lista de bloqueados
                document.getElementById('manual-block-ip-input').value = '';
                document.getElementById('manual-block-reason-input').value = '';
            }
        } else {
            Swal.fire('Atenção!', 'Por favor, insira um endereço IP para bloquear.', 'warning');
        }
    });

    // Botão Adicionar Whitelist
    document.getElementById('add-whitelist-btn').addEventListener('click', async () => {
        const ip = document.getElementById('whitelist-ip-input').value.trim();
        const notes = document.getElementById('whitelist-notes-input').value.trim();
        if (ip) {
            const result = await fetchData('allow_ip', { ip, notes });
             if(result) {
                Swal.fire(result.status === 'success' ? 'Sucesso!' : 'Erro!', result.message || 'Operação concluída.', result.status || 'info');
                loadLists();
                document.getElementById('whitelist-ip-input').value = '';
                document.getElementById('whitelist-notes-input').value = '';
            }
        } else {
            Swal.fire('Atenção!', 'Por favor, insira um endereço IP.', 'warning');
        }
    });
    
    // Botão Remover Whitelist
    document.getElementById('allowed-ips-table').addEventListener('click', async (e) => {
        if (e.target.classList.contains('remove-allowed-btn')) {
            const ip = e.target.dataset.ip;
            
            Swal.fire({
                title: 'Tem a certeza?',
                text: `Deseja realmente remover o IP ${ip} da whitelist?`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#d33',
                cancelButtonColor: '#3085d6',
                confirmButtonText: 'Sim, remover!',
                cancelButtonText: 'Cancelar'
            }).then(async (result) => {
                if (result.isConfirmed) {
                    const apiResult = await fetchData('remove_allowed', { ip });
                    if(apiResult) {
                        Swal.fire(apiResult.status === 'success' ? 'Removido!' : 'Erro!', apiResult.message || 'Operação concluída.', apiResult.status || 'info');
                        loadLists();
                    }
                }
            });
        }
    });
    
    document.getElementById('check-all-banned').addEventListener('change', (e) => {
        document.querySelectorAll('.banned-ip-check').forEach(cb => cb.checked = e.target.checked);
    });

    loadLists(); // Carrega as listas quando a página abre
});
</script>