// Arquivo: custom.js (ou o nome do seu arquivo JS principal)
// VERSAO COMPLETA E CORRIGIDA - Resolve conflito de atributos (data-copy-text vs data-raw-text)
// [CORREÇÃO TESTE MANUAL] - Corrigido o ID do modal na funcao enviardados

document.addEventListener("DOMContentLoaded", function() {
    // Funcao para garantir overflow na tabela DataTable
    function addOverflowAuto() {
        var wrapper = document.getElementById("data_table_wrapper");
        if (wrapper) {
            // DataTables 1.x vs 2.x usam classes diferentes, tentar os dois
            var layoutCell = wrapper.querySelector(".dataTables_scrollBody") || wrapper.querySelector(".dt-layout-table");
             if (layoutCell && !layoutCell.classList.contains('overflow-auto')) {
                  layoutCell.classList.add("overflow-auto");
                  console.log("Classe overflow-auto adicionada."); // Para debug
             }
        }
    }
    // Observador para aplicar a classe quando a tabela for redesenhada/recarregada
    var observer = new MutationObserver(function(mutations) {
        mutations.forEach(function(mutation) {
            if (mutation.type === 'childList' || mutation.type === 'subtree') {
                addOverflowAuto();
            }
        });
    });
    observer.observe(document.body, { childList: true, subtree: true });
    addOverflowAuto(); // Executa uma vez ao carregar

    // --- FUNCOES AUXILIARES PARA MODAIS DE INFORMACAO ---
    // (Colocadas dentro do DOMContentLoaded para garantir que os elementos do modal existem)

    window.copiarTextoModal = function(elementId) {
        const contentElement = document.getElementById(elementId);
        
        // ===================================================================
        // ### CORRECAO APLICADA AQUI ###
        // Procura primeiro por 'data-copy-text' (usado pelo app.js) 
        // e depois mantÃ©m o 'data-raw-text' como fallback.
        // ===================================================================
        const textToCopy = contentElement ? (contentElement.getAttribute('data-copy-text') || contentElement.getAttribute('data-raw-text')) : null;

        if (!textToCopy) {
             Swal.fire('Atencao!', 'Nenhum texto para copiar.', 'warning');
             return;
        }

        if (navigator.clipboard) {
            navigator.clipboard.writeText(textToCopy).then(() => {
                Swal.fire('Copiado!', 'Os dados foram copiados.', 'success');
            }).catch(err => {
                console.error('Erro ao copiar:', err);
                Swal.fire('Erro!', 'Nao foi possivel copiar.', 'error');
            });
        } else {
            // Fallback para navegadores mais antigos ou HTTP
            const textArea = document.createElement("textarea");
            textArea.value = textToCopy;
            textArea.style.position = 'fixed'; // Evita scroll
            textArea.style.left = '-9999px';
            document.body.appendChild(textArea);
            textArea.focus();
            textArea.select();
            try {
                document.execCommand('copy');
                Swal.fire('Copiado!', 'Os dados foram copiados.', 'success');
            } catch (err) {
                console.error('Erro ao copiar (fallback):', err);
                Swal.fire('Erro!', 'Nao foi possivel copiar.', 'error');
            }
            document.body.removeChild(textArea);
        }
    }

    window.enviarParaWhatsAppModal = function(elementId) {
        const contentElement = document.getElementById(elementId);
        
        // APLICANDO A MESMA CORRECAO AQUI PARA GARANTIR
        const textToSend = contentElement ? (contentElement.getAttribute('data-copy-text') || contentElement.getAttribute('data-raw-text')) : null;

        if (!textToSend) {
             Swal.fire('Atencao!', 'Nenhum texto para enviar.', 'warning');
             return;
        }

        const whatsappUrl = "https://api.whatsapp.com/send?text=" + encodeURIComponent(textToSend);
        window.open(whatsappUrl, '_blank');
    }

    // Funcao chamada pelo modal_master de info_cliente (mantem nome por compatibilidade)
    window.copiarMensagem = function(elementId) {
        copiarTextoModal(elementId); // Chama a funcao mais robusta
    }

}); // Fim do DOMContentLoaded


// --- FUNCOES GLOBAIS ---

// Funcao para atualizar creditos na interface
async function updateinfo() {
    try {
        // Endpoint padrao para buscar info do admin logado
        // Ajuste se o seu arquivo for diferente (ex: api/admin_info.php)
        const response = await fetch(`api/listar-clientes.php?info_admin`);
        if (!response.ok) { throw new Error('Erro na rede ao buscar info admin'); }

        const jsonResponse = await response.json();

        if (jsonResponse.icon === 'error') {
            // Nao mostrar alerta se for apenas erro de sessao ao carregar pagina
            if (jsonResponse.msg && !jsonResponse.msg.toLowerCase().includes('sessao')) {
               SweetAlert2(jsonResponse.title, jsonResponse.msg, jsonResponse.icon);
            } else {
               console.warn("Info Admin:", jsonResponse.msg); // Log silencioso
            }
        } else if (jsonResponse.creditos !== undefined && jsonResponse.tipo_admin !== undefined) {
            // Atualiza creditos em todos os lugares necessarios
            $('#creditos, #creditos2, #creditos-mobile').html(jsonResponse.creditos);
            $('#tipo_admin').html(jsonResponse.tipo_admin);
            // $('#vencimento').html(jsonResponse.vencimento); // Descomente e ajuste se houver vencimento
        } else {
            console.warn("Resposta info_admin invalida:", jsonResponse);
        }
    } catch (error) {
        console.error('Falha ao atualizar informacoes do admin:', error);
        // Evitar alerta em caso de falha de rede simples no carregamento inicial
        // SweetAlert2('Erro!', 'Nao foi possivel carregar dados do usuario.', 'error');
    }
}

// Chama a atualizacao de info ao carregar a pagina
updateinfo();


// Funcao para abrir modal generico vindo do PHP
async function modal_master(url, parametro1, valor1) {
    // Mostra loading enquanto carrega
     Swal.fire({
         title: 'Carregando...',
         allowOutsideClick: false,
         didOpen: () => { Swal.showLoading(); }
    });

    $.ajax({
        type: "POST",
        url: url, // O caminho completo (ex: api/clientes.php) ja vem no onclick
        data: { [parametro1]: valor1 },
        dataType: 'json',
        success: function(response) {
            Swal.close(); // Fecha o loading

            if (response && response.modal_titulo && response.modal_body && response.modal_footer) {
                $('#modal_master-titulo').html(response.modal_titulo);
                $('#modal_master-body').html(response.modal_body);
                $('#modal_master-footer').html(response.modal_footer);

                // Aplica classe de cor ao header do modal, se definida pelo PHP
                 const header = $('#modal_master .modal-header');
                 // Limpa classes de cor anteriores (bg-*) e text-white
                 header.removeClass (function (index, className) {
                     return (className.match (/(^|\s)bg-\S+/g) || []).join(' ');
                 }).removeClass('text-white');

                 if (response.modal_header_class) {
                     // Adiciona as classes especificadas pelo PHP (ex: "modal-header bg-success text-white")
                     const classes = response.modal_header_class.split(' ');
                     classes.forEach(cls => header.addClass(cls));
                 } else {
                     header.addClass('bg-primary text-white'); // Cor padrao
                 }

                // Usa getOrCreateInstance para evitar erros
                 const modalEl = document.getElementById('modal_master');
                 const modalInstance = bootstrap.Modal.getOrCreateInstance(modalEl);
                 modalInstance.show();

            } else if (response && response.icon && response.title && response.msg) {
                // Se o PHP retornar um erro direto (sem modal)
                SweetAlert2(response.title, response.msg, response.icon);
            } else {
                // Se a resposta for invalida
                 console.error("Resposta invalida do modal_master:", response);
                 SweetAlert2('Erro!', 'Resposta invalida do servidor ao carregar modal.', 'error');
            }
        },
        error: function(jqXHR, textStatus, errorThrown) {
            Swal.close(); // Fecha o loading
            console.error("AJAX Error (modal_master):", textStatus, errorThrown, jqXHR.responseText);
            SweetAlert2('Erro!', 'Erro de comunicacao ao carregar o modal.', 'error');
        }
    });
}


// Funcao principal para enviar formularios via AJAX
var solicitacaoPendente = false; // Flag para evitar cliques duplos
function enviardados(id_formulario, url_arquivo_php) {
    if (solicitacaoPendente) {
        SweetAlert3('Aguarde!', 'Solicitacao em andamento...', 'warning');
        return;
    }
    solicitacaoPendente = true;

    var formElement = document.getElementById(id_formulario);
    if (!formElement) {
        console.error("Formulario nao encontrado:", id_formulario);
        SweetAlert2('Erro Interno', 'Formulario nao encontrado.', 'error');
        solicitacaoPendente = false;
        return;
    }
    var formData = new FormData(formElement);

    // Mostra loading robusto
    Swal.fire({
        title: 'Aguarde...',
        text: 'Processando sua solicitacao.',
        allowOutsideClick: false,
        didOpen: () => { Swal.showLoading(); }
    });

    $.ajax({
        type: "POST",
        url: "api/" + url_arquivo_php, // Adiciona 'api/' ao nome do arquivo PHP
        data: formData,
        processData: false, // Necessario para FormData
        contentType: false, // Necessario para FormData
        dataType: 'json',
        success: function(res) {
            Swal.close(); // Fecha o loading antes de processar

            if (!res) {
                SweetAlert2('Erro!', 'Resposta do servidor vazia ou invalida.', 'error');
                return;
            }

            // Define se deve exibir modal de informacao apos sucesso
            const exibirModalInfo = res.icon === 'success' && (res.dados_cliente_msg || res.dados_operacao_msg || res.dados_revendedor_msg);

            // 1. Mostra o alerta (SweetAlert) padrao, exceto se for abrir modal de info
            if (res.title && res.msg && res.icon && !exibirModalInfo) {
                SweetAlert2(res.title, res.msg, res.icon);
            } else if (!res.title || !res.msg || !res.icon) {
                 console.warn("Resposta AJAX sem title/msg/icon:", res); // Avisa se faltar algo
            }

            // 2. Acoes em caso de SUCESSO
            if (res.icon === 'success') {
                // Fecha o modal do FORMULARIO (modal_master)
                const modalFormularioEl = document.getElementById('modal_master');
                if (modalFormularioEl) {
                    const modalFormularioInstance = bootstrap.Modal.getInstance(modalFormularioEl);
                    // Verifica se a instancia existe antes de tentar esconder
                    if (modalFormularioInstance) {
                         modalFormularioInstance.hide();
                    }
                }

                // Recarrega a DataTable principal (se existir e se PHP mandar)
                if (res.data_table === 'atualizar') {
                    try {
                        // Usa um seletor mais generico que busca qualquer DataTable inicializada na pagina
                        const dataTableInstance = $('.dataTable').DataTable();
                        if (dataTableInstance) {
                             dataTableInstance.ajax.reload(null, false); // false = nao reseta paginacao
                        } else {
                             console.warn("Nenhuma DataTable encontrada para recarregar.");
                        }
                    } catch (e) {
                         console.warn("Erro ao tentar recarregar DataTable:", e);
                    }
                }

                 // Redireciona se necessario
                 if (res.redirect_url) {
                     // Adiciona um pequeno delay para o usuario ver o alerta de sucesso
                     setTimeout(() => { window.location.href = res.redirect_url; }, 1000);
                     return;
                 }

                // Atualiza creditos na interface
                updateinfo();

                // --- LOGICA PARA EXIBIR MODAIS DE INFORMACAO ---
                let mensagemParaCopiar = null;
                let modalParaExibirId = null;
                let divConteudoId = null;

                if (res.dados_cliente_msg) { // Mensagens de Clientes ou Testes
                    mensagemParaCopiar = res.dados_cliente_msg;
                    // ===================================================================
                    // ### CORREÇÃO APLICADA AQUI ###
                    // O modal de info de cliente/teste deve ser 'infoClienteModal'
                    // (Assumindo que ele exista no seu HTML, assim como existe o de revendedor)
                    // Se o seu modal de cliente tiver outro ID, troque aqui.
                    // ===================================================================
                    modalParaExibirId = 'infoClienteModal'; 
                    divConteudoId = 'dadosClienteParaCopiar'; 

                } else if (res.dados_operacao_msg) { // Mensagem de Creditos
                    mensagemParaCopiar = res.dados_operacao_msg;
                    modalParaExibirId = 'infoCreditosModal';
                    divConteudoId = 'dadosCreditosParaCopiar';
                } else if (res.dados_revendedor_msg) { // Mensagem de Novo Revendedor
                    mensagemParaCopiar = res.dados_revendedor_msg;
                    modalParaExibirId = 'infoRevendedorModal';
                    divConteudoId = 'dadosRevendedorParaCopiar';
                }

                // Se encontrou uma mensagem, exibe o modal correspondente
                if (exibirModalInfo && mensagemParaCopiar && modalParaExibirId && divConteudoId) {
                    const modalElement = document.getElementById(modalParaExibirId);
                    const contentElement = document.getElementById(divConteudoId);

                    if (modalElement && contentElement) {
                        // Usa innerText para que \n vire quebra de linha visualmente no modal
                        contentElement.innerText = mensagemParaCopiar;
                        
                        // ===================================================================
                        // ### CORRECAO DE COMPATIBILIDADE ###
                        // Define *ambos* os atributos para funcionar com custom.js e app.js
                        // ===================================================================
                        contentElement.setAttribute('data-raw-text', mensagemParaCopiar);
                        contentElement.setAttribute('data-copy-text', mensagemParaCopiar);


                        const modalInstance = bootstrap.Modal.getOrCreateInstance(modalElement);
                        modalInstance.show();
                    } else {
                        console.error('Modal de copia/info ou div de conteudo nao encontrado:', modalParaExibirId, divConteudoId);
                        SweetAlert2('Erro Interno', 'Nao foi possivel exibir os dados para copiar.', 'error');
                    }
                }
                // --- FIM DA LOGICA DOS MODAIS DE INFORMACAO ---
            } // Fim if (res.icon === 'success')

        }, // Fim success
        error: function(jqXHR, textStatus, errorThrown) {
            Swal.close(); // Fecha o loading
            console.error("AJAX Error (enviardados):", textStatus, errorThrown, jqXHR.responseText);
            // Tenta mostrar erro do servidor se disponivel
             let errorMsg = 'Erro na solicitacao ao servidor.';
             if (jqXHR.responseJSON && jqXHR.responseJSON.msg) {
                 errorMsg = jqXHR.responseJSON.msg;
             } else if (jqXHR.responseText) {
                 // Evita mostrar HTML de erro completo, pega so a primeira linha
                 try {
                     const parser = new DOMParser();
                     const doc = parser.parseFromString(jqXHR.responseText, "text/html");
                     const bodyText = doc.body.textContent || "";
                     errorMsg = bodyText.split('\n')[0].trim() || errorMsg;
                 } catch (e) { /* Ignora erro de parse */ }
             }
            SweetAlert2('Erro!', errorMsg, 'error');
        },
        complete: function() {
            solicitacaoPendente = false; // Libera para novas requisicoes
        }
    }); // Fim $.ajax
} // Fim enviardados


// Funcao padrao para SweetAlert2 (mantida)
function SweetAlert2(title, text, icon) {
    Swal.fire({ title: title, html: text, icon: icon });
}

// Funcao padrao para SweetAlert3 (Toast) (mantida)
function SweetAlert3(title, icon, timer) {
    const Toast = Swal.mixin({
        toast: true,
        position: "top-end",
        showConfirmButton: false,
        timer: timer || 3000,
        timerProgressBar: true,
        didOpen: (toast) => {
            toast.onmouseenter = Swal.stopTimer;
            toast.onmouseleave = Swal.resumeTimer;
        }
    });
    Toast.fire({ icon: icon, title: title });
}