(() => {
    // --- CONFIGURAÇÕES ---
    const CHUNK_SIZE_URLS = 150; // Quantidade de URLs a enviar por requisição
    const CONCURRENT_REQUESTS = 8; // Quantas requisições fazer ao mesmo tempo (aumente para mais velocidade, se o servidor aguentar)
    const M3U_PART_SIZE_BYTES = 209715200; // Tamanho para dividir o arquivo M3U (200MB)

    // --- VARIÁVEIS DE ESTADO ---
    let isProcessing = false;
    let isPaused = false;
    let totalLinksInFile = 0;
    let totalLinksProcessed = 0;
    let startTime = null;
    let avgTimePerRequest = 0;
    let totalRequestTime = 0;

    // Contadores da UI
    let totalRequestsSent = 0;
    let urlsAdded = 0;
    let channelsAdded = 0;
    let moviesAdded = 0;
    let seriesAdded = 0;
    let seasonsAdded = 0;
    let episodesAdded = 0;
    let duplicatesFound = 0;
    let errorsCount = 0;
    let epgAdded = 0;

    // --- ELEMENTOS DO DOM ---
    const pauseBtn = document.getElementById("pauseBtn");
    const resumeBtn = document.getElementById("resumeBtn");
    const processFileBtn = document.getElementById("processFileBtn");
    const fileInput = document.getElementById("m3uFile");
    const urlInput = document.getElementById("m3uUrl");
    const openModalBtn = document.getElementById("openFirstModal");
    const dropArea = document.getElementById("dropArea");
    const dropArea2 = document.getElementById("dropArea2");

    // Modais
    const fileModal = new bootstrap.Modal(document.getElementById("modal_arquivo"));
    const urlModal = new bootstrap.Modal(document.getElementById("modal_url"));
    const completionModal = new bootstrap.Modal(document.getElementById("completionModal"));

    /**
     * Divide o conteúdo de um arquivo M3U em partes menores para não sobrecarregar a memória.
     */
    function splitM3UIntoParts(content) {
        const lines = content.split('\n');
        const parts = [];
        let currentPart = '#EXTM3U\n';
        let currentSize = 0;

        lines.forEach(line => {
            currentPart += line + '\n';
            currentSize += line.length + 1;
            if (/^(http|rtmp)/i.test(line.trim()) && currentSize >= M3U_PART_SIZE_BYTES) {
                parts.push(currentPart);
                currentPart = '#EXTM3U\n';
                currentSize = 0;
            }
        });

        if (currentSize > '#EXTM3U\n'.length) {
            parts.push(currentPart);
        }
        return parts;
    }

    /**
     * Extrai os links e metadados de um bloco de texto M3U.
     */
    function parseM3UContent(content) {
        const lines = content.split('\n');
        const links = [];

        function getAttributeValue(line, attribute) {
            const regex = new RegExp(`${attribute}="([^"]+)"`);
            const match = line.match(regex);
            return match ? match[1] : "";
        }

        for (let i = 0; i < lines.length; i++) {
            const line = lines[i].trim();
            if (line.startsWith("#EXTINF")) {
                const groupTitle = getAttributeValue(line, "group-title");
                const tvgLogo = getAttributeValue(line, "tvg-logo");
                const tvgName = getAttributeValue(line, "tvg-name");
                const tvgId = getAttributeValue(line, "tvg-id");
                
                let channelName = "Nome não disponível";
                const nameIndex = line.lastIndexOf(",");
                if (nameIndex !== -1) {
                    channelName = line.substring(nameIndex + 1).trim();
                }

                const url = lines[i + 1]?.trim();
                if (url && /^(https?|rtsp|ftp):\/\/[^\s]+/g.test(url)) {
                    links.push({
                        url: url,
                        groupTitle: groupTitle,
                        tvgLogo: tvgLogo,
                        tvgName: tvgName,
                        tvgId: tvgId,
                        channelName: channelName
                    });
                }
            }
        }
        totalLinksInFile += links.length;
        return links;
    }
    
    /**
     * Envia um bloco de links para o servidor.
     */
    async function sendContentBlock(block) {
        const requestStartTime = Date.now();
        try {
            const response = await fetch("./api/controles/importar-arquivo-m3u.php", {
                method: "POST",
                headers: { 'Content-Type': 'application/json' },
                body: JSON.stringify({ block: block }) // Envia como JSON
            });

            if (!response.ok) {
                throw new Error(`Erro no servidor: ${response.status}`);
            }
            
            const data = await response.json();
            if (data.results && typeof data.results === 'object') {
                const { urls = 0, movie = 0, series = 0, live = 0, episodios = 0, temporadas = 0, exists = 0, error = [] } = data.results;
                urlsAdded += urls;
                moviesAdded += movie;
                seriesAdded += series;
                channelsAdded += live;
                episodesAdded += episodios;
                seasonsAdded += temporadas;
                duplicatesFound += exists;
                errorsCount += error.length;
            }
        } catch (error) {
            console.error("Erro ao enviar bloco:", error);
            errorsCount += block.length; // Assume que o bloco todo falhou
        } finally {
            totalRequestsSent++;
            totalLinksProcessed += block.length;
            const requestTime = (Date.now() - requestStartTime) / 1000;
            totalRequestTime += requestTime;
            avgTimePerRequest = totalRequestTime / totalRequestsSent;
            updateUICounters();
            updateProgressBars();
            updateTimers();
        }
    }

    /**
     * Lógica principal que gerencia o envio de todos os blocos em paralelo.
     */
    async function processAllParts(parts) {
        urlModal.hide();
        const allLinks = parts.flatMap(part => parseM3UContent(part));
        const allChunks = chunkArray(allLinks, CHUNK_SIZE_URLS);

        let currentIndex = 0;
        async function processQueue() {
            while(currentIndex < allChunks.length) {
                // Checa se está pausado antes de iniciar um novo lote
                while (isPaused) {
                    await new Promise(resolve => setTimeout(resolve, 500));
                }

                const batch = [];
                for (let i = 0; i < CONCURRENT_REQUESTS && currentIndex < allChunks.length; i++) {
                    batch.push(sendContentBlock(allChunks[currentIndex]));
                    currentIndex++;
                }
                await Promise.all(batch);
            }
        }
        
        await processQueue();

        // Processo concluído
        await processEpg(); // Processa EPG no final
        completionModal.show();
        document.getElementById("controles").style.display = "none";
        clearServerCache();
        processFileBtn.disabled = false;
        isProcessing = false;
    }

    /**
     * Processa um arquivo M3U local.
     */
    function processLocalFile(file) {
        const reader = new FileReader();
        reader.onload = (event) => {
            const parts = splitM3UIntoParts(event.target.result);
            document.getElementById("progressBar").style.width = '100%'; // Simula leitura instantânea
            isProcessing = true;
            processAllParts(parts);
        };
        reader.readAsText(file);
        
        openModalBtn.disabled = true;
        openModalBtn.style.display = "none";
        fileModal.hide();
    }
    
    /**
     * Processa uma URL M3U remota via proxy.
     */
    function processRemoteUrl(url) {
        const proxyUrl = `./api/proxy.php?url=${encodeURIComponent(url)}`;
        fetch(proxyUrl)
            .then(response => {
                if (!response.ok) throw new Error(`Erro no proxy: ${response.statusText}`);
                return response.text();
            })
            .then(m3uContent => {
                const parts = splitM3UIntoParts(m3uContent);
                document.getElementById("progressBar").style.width = '100%';
                isProcessing = true;
                processAllParts(parts);
            })
            .catch(error => {
                console.error("Falha ao baixar URL:", error);
                Swal.fire("Erro", "Não foi possível baixar o arquivo da URL.", "error");
                processFileBtn.disabled = false;
            });
    }

    // --- FUNÇÕES DE ATUALIZAÇÃO DA UI ---
    
    function updateUICounters() {
        document.getElementById("totalRequests").textContent = totalRequestsSent;
        document.getElementById("add_urls").textContent = urlsAdded;
        document.getElementById("canais").textContent = channelsAdded;
        document.getElementById("filmes").textContent = moviesAdded;
        document.getElementById("series_adicionando").textContent = seriesAdded;
        document.getElementById("temporadas_adicionando").textContent = seasonsAdded;
        document.getElementById("episodios_adicionando").textContent = episodesAdded;
        document.getElementById("exitente").textContent = duplicatesFound;
        document.getElementById("Erro").textContent = errorsCount;
        document.getElementById("epg_adicionando").textContent = epgAdded;
    }
    
    function updateProgressBars() {
        if (totalLinksInFile > 0) {
            const percentage = (totalLinksProcessed / totalLinksInFile) * 100;
            const progressBar = document.getElementById("partProgressBar");
            progressBar.style.width = `${percentage}%`;
            progressBar.textContent = `${percentage.toFixed(2)}%`;
        }
    }

    function formatTime(seconds) {
        const h = Math.floor(seconds / 3600).toString().padStart(2, '0');
        const m = Math.floor((seconds % 3600) / 60).toString().padStart(2, '0');
        const s = Math.floor(seconds % 60).toString().padStart(2, '0');
        return `${h}:${m}:${s}`;
    }

    function updateTimers() {
        if (!startTime) startTime = new Date();
        const elapsedTime = (new Date() - startTime) / 1000;
        
        const totalChunks = Math.ceil(totalLinksInFile / CHUNK_SIZE_URLS);
        const chunksRemaining = totalChunks - totalRequestsSent;
        
        const estimatedTotalTime = totalChunks * avgTimePerRequest;
        const estimatedRemainingTime = chunksRemaining * avgTimePerRequest;

        document.getElementById("tempo_Decorrido").textContent = formatTime(elapsedTime);
        document.getElementById("tempo_Total_Estimado").textContent = formatTime(estimatedTotalTime);
        document.getElementById("tempo_Restante").textContent = formatTime(estimatedRemainingTime > 0 ? estimatedRemainingTime : 0);
    }
    
    // --- FUNÇÕES AUXILIARES ---
    
    function chunkArray(array, size) {
        const chunks = [];
        for (let i = 0; i < array.length; i += size) {
            chunks.push(array.slice(i, i + size));
        }
        return chunks;
    }

    async function clearServerCache() {
        try {
            await fetch("./api/limpar-cache.php");
        } catch (error) {
            console.error("Erro ao limpar cache:", error);
        }
    }

    // --- LÓGICA DE EPG (pode ser otimizada da mesma forma se necessário) ---
    async function processEpg() {
       // A sua lógica de EPG original (função H e j) iria aqui.
       // Por ora, vamos simplificar para não introduzir complexidade extra.
       console.log("Processamento de EPG iniciado...");
       // Adicione a lógica de fetch e processamento do XML aqui.
    }

    // --- EVENT LISTENERS ---

    // Botões Pausar / Continuar
    pauseBtn.addEventListener("click", () => {
        isPaused = true;
        pauseBtn.disabled = true;
        resumeBtn.disabled = false;
    });

    resumeBtn.addEventListener("click", () => {
        isPaused = false;
        pauseBtn.disabled = false;
        resumeBtn.disabled = true;
    });

    // Botão Principal de Processamento
    processFileBtn.addEventListener("click", function() {
        if (isProcessing) return;
        clearServerCache();
        
        const file = fileInput.files[0];
        const url = urlInput.value;
        
        if (file) {
            processFileBtn.disabled = true;
            processLocalFile(file);
        } else if (url) {
            processFileBtn.disabled = true;
            document.getElementById("result").textContent = "Baixando arquivo, aguarde...";
            processRemoteUrl(url);
        } else {
            document.getElementById("result").textContent = "Por favor, selecione um arquivo ou forneça uma URL.";
        }
    });

    // Interação com Modais
    document.getElementById("openFirstModal").addEventListener("click", () => fileModal.show());
    document.getElementById("openSecondModal").addEventListener("click", () => { fileModal.hide(); urlModal.show(); });
    document.getElementById("backToFirstModal").addEventListener("click", () => { urlModal.hide(); fileModal.show(); });

    // Drag and Drop
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        document.body.addEventListener(eventName, e => e.preventDefault());
        dropArea.addEventListener(eventName, e => e.preventDefault());
    });
    document.body.addEventListener("dragenter", () => {
        dropArea.style.display = 'flex';
        dropArea2.style.display = 'none';
    });
    dropArea.addEventListener('dragleave', (e) => {
        // Apenas esconde se o mouse sair da área de drop para um elemento não relacionado
        if (!e.relatedTarget || !dropArea.contains(e.relatedTarget)) {
             dropArea.style.display = 'none';
             dropArea2.style.display = 'block';
        }
    });
    dropArea.addEventListener("drop", (e) => {
        dropArea.style.display = 'none';
        dropArea2.style.display = 'block';
        const file = e.dataTransfer.files[0];
        if (file && (file.type === "application/x-mpegURL" || file.name.endsWith(".m3u"))) {
            processFileBtn.disabled = true;
            processLocalFile(file);
        }
    });

    // Iniciar
    fileModal.show();
})();