/**
 * uploud2026.js
 * [CORREO] Adicionado um log de debug para ver o que o 'fileContent' realmente contm.
 */

// ===================================================================
//                VARIVEIS GLOBAIS DE CONTROLE
// ===================================================================
let fileContent = "";
let parts = [];
let totalParts = 0;
let completedParts = 0; 
let pauseProcessing = false;
let startTime = null;
let elapsedTimeUpdater = null;
let remainingTimeUpdater = null;
let useTmdb = true;
let categoriesToImport = [];
const CONCURRENT_REQUESTS = 5;
let activeWorkers = 0;
let importedContentReport = []; 
const counters = {
    add_urls: 0,
    canais: 0,
    filmes: 0,
    series_adicionando: 0,
    temporadas_adicionando: 0,
    episodios_adicionando: 0,
    exitente: 0,
    Erro: 0,
    epg_adicionando: 0
};

// ===================================================================
//                INICIALIZAO (DOM LOADED)
// ===================================================================
document.addEventListener("DOMContentLoaded", () => {
    // --- Modais
    const modalArquivo = new bootstrap.Modal(document.getElementById('modal_arquivo'));
    const modalUrl = new bootstrap.Modal(document.getElementById('modal_url'));
    const modalCategorias = new bootstrap.Modal(document.getElementById('modal_categorias_tmdb')); 
    const completionModal = new bootstrap.Modal(document.getElementById('completionModal')); 

    // --- Botes
    document.getElementById('openFirstModal').addEventListener('click', () => modalArquivo.show());
    document.getElementById('btnSelectFile').addEventListener('click', () => document.getElementById('m3uFile').click());
    document.getElementById('openSecondModal').addEventListener('click', () => {
        modalArquivo.hide();
        modalUrl.show();
    });
    document.getElementById('backToFirstModal').addEventListener('click', () => {
        modalUrl.hide();
        modalArquivo.show();
    });
    document.getElementById('pauseBtn').addEventListener('click', pause);
    document.getElementById('resumeBtn').addEventListener('click', resume);
    document.getElementById('processFileBtn').addEventListener('click', processUrlFile);
    document.getElementById('startImportBtn').addEventListener('click', () => {
        categoriesToImport = getSelectedCategories();
        useTmdb = document.getElementById('useTmdb').checked;
        modalCategorias.hide();
        startProcessing();
    });

    // --- Upload de Arquivo
    document.getElementById('m3uFile').addEventListener('change', handleFileSelect);

    // --- Drag and Drop
    const dropArea = document.getElementById('dropArea');
    const dropArea2 = document.getElementById('dropArea2');
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => {
        dropArea.addEventListener(eventName, preventDefaults, false);
        dropArea2.addEventListener(eventName, preventDefaults, false);
    });
    ['dragenter', 'dragover'].forEach(eventName => {
        dropArea.addEventListener(eventName, () => dropArea.style.display = 'block', false);
        dropArea2.addEventListener(eventName, highlight, false);
    });
    ['dragleave', 'drop'].forEach(eventName => {
        dropArea.addEventListener(eventName, () => dropArea.style.display = 'none', false);
        dropArea2.addEventListener(eventName, unhighlight, false);
    });
    dropArea.addEventListener('drop', handleDrop, false);
    dropArea2.addEventListener('drop', handleDrop, false);

    // --- Seleo de Categoria
    document.getElementById('selectAllCategories').addEventListener('change', (e) => {
        document.querySelectorAll('#categoryList .form-check-input').forEach(checkbox => {
            checkbox.checked = e.target.checked;
        });
    });
});

function preventDefaults(e) { e.preventDefault(); e.stopPropagation(); }
function highlight() { document.getElementById('dropArea2').classList.add('bg-light'); }
function unhighlight() { document.getElementById('dropArea2').classList.remove('bg-light'); }

function handleDrop(e) {
    const dt = e.dataTransfer;
    const file = dt.files[0];
    if (file) {
        document.getElementById('m3uFile').files = dt.files;
        handleFileSelect({ target: { files: [file] } });
    }
}

// ===================================================================
//                PASSO 1: LEITURA DO ARQUIVO (URL ou Local)
// ===================================================================

function handleFileSelect(evt) {
    const file = evt.target.files[0];
    if (file) {
        const reader = new FileReader();
        reader.onload = function(e) {
            fileContent = e.target.result; // Vamos deixar o parseM3U limpar
            bootstrap.Modal.getInstance(document.getElementById('modal_arquivo')).hide();
            parseM3U();
        };
        reader.readAsText(file);
    }
}

async function processUrlFile() {
    const url = document.getElementById('m3uUrl').value;
    if (!url) {
        Swal.fire('Erro', 'Por favor, insira uma URL vlida.', 'error');
        return;
    }

    addLog('info', `Baixando arquivo da URL: ${url}`);
    try {
        const response = await fetch('api/fetch_url.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json', },
            body: JSON.stringify({ url: url })
        });

        const responseText = await response.text();
        let result;
        try {
            result = JSON.parse(responseText);
        } catch (e) {
            throw new Error(responseText); 
        }

        if (!response.ok || result.error) {
            throw new Error(result.error || `Erro de servidor: ${response.statusText}`);
        }
        
        fileContent = result.content; // Pega o contedo (que o PHP j limpou)
        bootstrap.Modal.getInstance(document.getElementById('modal_url')).hide();
        addLog('success', 'Arquivo baixado com sucesso. Analisando...');
        parseM3U();
    } catch (error) {
        addLog('error', `Falha ao processar URL: ${error.message}`);
        Swal.fire('Erro', `Falha ao processar URL: ${error.message}`, 'error');
    }
}

// ===================================================================
//                PASSO 2: ANLISE DO M3U (S CATEGORIAS)
// ===================================================================

function parseM3U() {
    
    // --- CORREO M3U INVLIDO (JavaScript) ---
    // Limpa o contedo (remove BOM e espaos em branco do incio)
    // Isso  uma garantia extra caso a limpeza do PHP falhe.
    if (fileContent) {
        fileContent = fileContent.replace(/^\xEF\xBB\xBF/, '').trimStart();
    }
    // --- FIM DA CORREO ---

    
    // ==================================================
    //         PASSO DE DEBUG (TEMPORRIO)
    // ==================================================
    // Vamos registrar as primeiras 200 letras do que recebemos
    console.log("Contedo recebido (primeiros 200 caracteres):");
    console.log(fileContent.substring(0, 200));
    addLog('info', 'DEBUG: Incio do arquivo recebido: ' + fileContent.substring(0, 200));
    // ==================================================


    if (!fileContent || !fileContent.startsWith('#EXTM3U')) {
        addLog('error', 'Arquivo M3U invlido ou mal formatado. O contedo (veja log acima) NO comea com #EXTM3U.');
        Swal.fire('Erro', 'Arquivo M3U invlido. O arquivo deve comear com #EXTM3U.', 'error');
        return;
    }

    addLog('info', 'Analisando arquivo M3U para encontrar categorias...');
    
    parts = []; 
    resetCounters(); 
    importedContentReport = []; 
    
    const lines = fileContent.split('\n');
    const categories = new Set();
    let currentCategory = null;

    for (let i = 0; i < lines.length; i++) {
        const line = lines[i].trim();

        if (line.startsWith('#EXTINF:')) {
            const attributes = line.match(/([a-zA-Z0-9_-]+)="([^"]*)"/g) || [];
            currentCategory = null; 
            attributes.forEach(attr => {
                const [key, value] = attr.replace(/"/g, '').split('=');
                if (key === 'group-title') {
                    currentCategory = value;
                }
            });
        } else if (line.startsWith('http://') || line.startsWith('https://') || line.startsWith('rtmp://')) {
            if (currentCategory) {
                categories.add(currentCategory);
                currentCategory = null; 
            }
        }
    }

    addLog('success', `Anlise concluda. ${categories.size} categorias encontradas.`);
    document.getElementById('partCount').textContent = `Encontradas ${categories.size} categorias.`;

    populateCategoryModal(categories);
}

function populateCategoryModal(categories) {
    const categoryList = document.getElementById('categoryList');
    categoryList.innerHTML = ''; 
    
    [...categories].sort().forEach(category => {
        const div = document.createElement('div');
        div.className = 'form-check';
        div.innerHTML = `
            <input class="form-check-input category-checkbox" type="checkbox" value="${category}" id="cat-${category.replace(/\s/g, '-')}" checked>
            <label class="form-check-label" for="cat-${category.replace(/\s/g, '-')}">
                ${category}
            </label>
        `;
        categoryList.appendChild(div);
    });

    const modal = bootstrap.Modal.getOrCreateInstance(document.getElementById('modal_categorias_tmdb'));
    modal.show();
}

function getSelectedCategories() {
    const selected = [];
    document.querySelectorAll('#categoryList .form-check-input:checked').forEach(checkbox => {
        selected.push(checkbox.value);
    });
    return selected;
}

// ===================================================================
//                ? NOVO: PASSO 2.5 - CONSTRUIR LOTES (APS SELEO)
// ===================================================================

function buildPartsList() {
    addLog('info', 'Filtrando arquivo M3U com as categorias selecionadas...');
    
    parts = []; 
    let tempPart = [];
    const partSize = 1000;
    
    const lines = fileContent.split('\n');
    let currentData = {};
    let itemsFound = 0;

    for (let i = 0; i < lines.length; i++) {
        const line = lines[i].trim();

        if (line.startsWith('#EXTINF:')) {
            currentData = {};
            const nameMatch = line.match(/,(.*)$/);
            currentData.name = nameMatch ? nameMatch[1].trim() : 'Nome no encontrado';

            const attributes = line.match(/([a-zA-Z0-9_-]+)="([^"]*)"/g) || [];
            attributes.forEach(attr => {
                const [key, value] = attr.replace(/"/g, '').split('=');
                if (key === 'tvg-id') currentData.tvgId = value;
                if (key === 'tvg-name') currentData.tvgName = value;
                if (key === 'tvg-logo') currentData.tvgLogo = value;
                if (key === 'group-title') currentData.category = value;
            });
            
            if (currentData.name.match(/[Ss]\d{1,2}[\s._-]?[EeXx\s._-]?\d{1,3}/i) || (currentData.category && currentData.category.match(/serie|srie/i))) {
                currentData.type = 'series';
            } else if (currentData.category && currentData.category.match(/filme|movie/i)) {
                currentData.type = 'movie';
            } else {
                currentData.type = 'live';
            }

        } else if (line.startsWith('http://') || line.startsWith('https://') || line.startsWith('rtmp://')) {
            if (currentData.name) {
                currentData.url = line;
                
                if (categoriesToImport.includes(currentData.category)) {
                    tempPart.push(currentData);
                    itemsFound++;
                    
                    if (tempPart.length >= partSize) {
                        parts.push(tempPart);
                        tempPart = [];
                    }
                }
                currentData = {}; // Reseta
            }
        }
    }

    if (tempPart.length > 0) {
        parts.push(tempPart);
    }

    totalParts = parts.length;
    completedParts = 0;
    
    addLog('success', `Filtro concludo. ${itemsFound} itens selecionados foram divididos em ${totalParts} lotes.`);
    document.getElementById('partCount').textContent = `Processando ${totalParts} lotes...`;
    updatePartProgressBar(); // Atualiza a UI
}


// ===================================================================
//                PASSO 3: PROCESSAMENTO DOS LOTES
// ===================================================================

function startProcessing() {
    categoriesToImport = getSelectedCategories(); 
    useTmdb = document.getElementById('useTmdb').checked;

    buildPartsList(); 

    if (parts.length === 0) {
        addLog('warning', 'Nenhum item encontrado para as categorias selecionadas. Nada a processar.');
        return;
    }

    addLog('info', `Iniciando importao de ${totalParts} lotes...`);
    
    window.addEventListener('beforeunload', showExitWarning);
    
    document.getElementById('pauseBtn').disabled = false;
    document.getElementById('resumeBtn').disabled = true;
    pauseProcessing = false;
    startTime = new Date();

    startTimers();
    
    activeWorkers = CONCURRENT_REQUESTS;
    for (let i = 0; i < CONCURRENT_REQUESTS; i++) {
        processBatch(i);
    }
}

async function processBatch(batchIndex) {
    if (batchIndex >= totalParts) {
        workerDone(); 
        return;
    }

    if (pauseProcessing) {
        await delay(1000); 
        processBatch(batchIndex); 
        return;
    }

    const partData = parts[batchIndex];

    try {
        addLog('info', `Processando lote ${batchIndex + 1} de ${totalParts} (contm ${partData.length} itens)...`);
        
        const response = await fetch('api/controles/importer2026_m3u.php', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json', },
            body: JSON.stringify({
                part: partData,
                useTmdb: useTmdb,
                totalParts: totalParts,
                currentPart: batchIndex
            })
        });

        if (!response.ok) {
            const errorText = await response.text();
            addLog('error', `Falha ao processar lote ${batchIndex + 1}: ${response.statusText}. Resposta do servidor:`);
            addLog('error', errorText); 
            throw new Error(`Erro de servidor no lote ${batchIndex + 1}: ${response.statusText}`);
        }

        const result = await response.json();
        processBatchResult(result, partData);

    } catch (error) {
        if (!error.message.includes('Erro de servidor')) {
            addLog('error', `Falha ao processar lote ${batchIndex + 1}: ${error.message}`);
        }
        counters.Erro += partData.length;
        updateCounterUI('Erro', counters.Erro);
    } finally {
        completedParts++;
        updatePartProgressBar();
        processBatch(batchIndex + CONCURRENT_REQUESTS);
    }
}

function workerDone() {
    activeWorkers--;
    if (activeWorkers === 0) {
        addLog('success', 'Processamento de todos os lotes concludo!');
        finishProcessing();
    }
}

function delay(ms) {
    return new Promise(resolve => setTimeout(resolve, ms));
}

function processBatchResult(result, partData) {
    if (result.logs) {
        result.logs.forEach(log => addLog(log.type, log.message));
    }

    if (result.counters) {
        for (const key in result.counters) {
            if (counters.hasOwnProperty(key)) {
                counters[key] += result.counters[key];
                updateCounterUI(key, counters[key]);
            }
        }
    }
    
    if (result.importedItems && Array.isArray(result.importedItems)) {
        result.importedItems.forEach(importedName => {
            const originalItem = partData.find(item => item.name === importedName);
            if (originalItem) {
                const type = originalItem.type || 'movie'; 
                if (type === 'movie' || type === 'series') {
                    const yearMatch = originalItem.name.match(/\((\d{4})\)/);
                    const year = yearMatch ? yearMatch[1] : '';
                    
                    importedContentReport.push({
                        name: originalItem.name,
                        category: originalItem.category || 'Sem Categoria',
                        type: type,
                        year: year
                    });
                }
            }
        });
    }
}


function finishProcessing() {
    window.removeEventListener('beforeunload', showExitWarning);
    
    if (elapsedTimeUpdater) clearInterval(elapsedTimeUpdater);
    if (remainingTimeUpdater) clearInterval(remainingTimeUpdater);
    document.getElementById('tempo_Restante').textContent = '00:00:00';

    const completionModal = bootstrap.Modal.getInstance(document.getElementById('completionModal')) || new bootstrap.Modal(document.getElementById('completionModal'));
    completionModal.show();
    
    document.getElementById('pauseBtn').disabled = true;
    document.getElementById('resumeBtn').disabled = true;
}

function showExitWarning(e) {
    const confirmationMessage = 'A importao est em andamento. Se voc sair agora, ela ser cancelada. Tem certeza?';
    e.preventDefault();
    e.returnValue = confirmationMessage; 
    return confirmationMessage;     
}

// ===================================================================
//                UI: PAUSA, LOGS, BARRAS E TIMERS
// ===================================================================

function pause() {
    pauseProcessing = true;
    document.getElementById('pauseBtn').disabled = true;
    document.getElementById('resumeBtn').disabled = false;
    if (elapsedTimeUpdater) clearInterval(elapsedTimeUpdater);
    if (remainingTimeUpdater) clearInterval(remainingTimeUpdater);
}

function resume() {
    pauseProcessing = false;
    document.getElementById('pauseBtn').disabled = false;
    document.getElementById('resumeBtn').disabled = true;
    
    startTime = new Date(new Date() - (completedParts / totalParts) * (new Date() - startTime)); 
    startTimers();
    activeWorkers = CONCURRENT_REQUESTS;
    for (let i = 0; i < CONCURRENT_REQUESTS; i++) {
        processBatch(i);
    }
}

function addLog(type, message) {
    const logContainer = document.getElementById('log-container');
    const logMessage = document.createElement('div');
    logMessage.className = `log-message ${type}`;
    logMessage.textContent = `[${new Date().toLocaleTimeString()}] [${type.toUpperCase()}] ${message}`;
    logContainer.appendChild(logMessage);
    logContainer.scrollTop = logContainer.scrollHeight; // Auto-scroll
}

function updateCounterUI(id, value) {
    const el = document.getElementById(id);
    if (el) {
        el.textContent = value;
    }
}

function updatePartProgressBar() {
    const percentage = totalParts > 0 ? Math.round((completedParts / totalParts) * 100) : 0;
    const progressBar = document.getElementById('partProgressBar');
    progressBar.style.width = `${percentage}%`;
    progressBar.textContent = `${percentage}%`;
    progressBar.setAttribute('aria-valuenow', percentage);

    document.getElementById('totalRequests').textContent = `${completedParts} / ${totalParts}`;
}

function resetCounters() {
    for (const key in counters) {
        counters[key] = 0;
        updateCounterUI(key, 0);
    }
    completedParts = 0; 
    updatePartProgressBar();
    document.getElementById('totalRequests').textContent = '0';
    document.getElementById('log-container').innerHTML = '';
    document.getElementById('tempo_Total_Estimado').textContent = '00:00:00';
    document.getElementById('tempo_Decorrido').textContent = '00:00:00';
    document.getElementById('tempo_Restante').textContent = '00:00:00';
}

function startTimers() {
    const estimatedTimePerBatch = 20000; // 20 segundos por lote
    const estimatedTotalTime = (totalParts / CONCURRENT_REQUESTS) * (estimatedTimePerBatch + 1000); 
    
    document.getElementById('tempo_Total_Estimado').textContent = formatTime(estimatedTotalTime);

    elapsedTimeUpdater = setInterval(() => {
        const elapsed = new Date() - startTime;
        document.getElementById('tempo_Decorrido').textContent = formatTime(elapsed);
    }, 1000);

    remainingTimeUpdater = setInterval(() => {
        const elapsed = new Date() - startTime;
        const progress = completedParts / totalParts; 
        const estimatedRemaining = progress > 0 ? (elapsed / progress) - elapsed : estimatedTotalTime;
        document.getElementById('tempo_Restante').textContent = formatTime(Math.max(0, estimatedRemaining));
    }, 1000);
}

function formatTime(ms) {
    const seconds = Math.floor((ms / 1000) % 60);
    const minutes = Math.floor((ms / (1000 * 60)) % 60);
    const hours = Math.floor((ms / (1000 * 60 * 60)) % 24);
    return [hours, minutes, seconds].map(v => v.toString().padStart(2, '0')).join(':');
}