#!/bin/bash

# ==============================================================================
#  Script de Instalação INTERATIVO para Load Balancer de Painel IPTV
# ==============================================================================
#  Este script irá solicitar as informações necessárias antes de iniciar
#  a instalação e configuração do Nginx, PHP-FPM e Firewall (UFW).
# ==============================================================================

# Cores para as mensagens
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
CYAN='\033[0;36m'
RED='\033[0;31m'
NC='\033[0m' # No Color

# Faz o script parar se algum comando falhar
set -e

# Verifica se o script está sendo executado como root
if [ "$(id -u)" -ne 0 ]; then
   echo -e "${RED}ERRO: Este script precisa ser executado como root. Use 'sudo ./install.sh'${NC}"
   exit 1
fi

clear
echo -e "${GREEN}=====================================================${NC}"
echo -e "${YELLOW}Instalador Interativo para Load Balancer IPTV${NC}"
echo -e "${GREEN}=====================================================${NC}"
echo "Este script irá configurar um novo servidor de Load Balancer."
echo "Por favor, insira as informações solicitadas a seguir."
echo ""

# --- COLETA DE DADOS DO USUÁRIO ---

# Detecta o IP principal da máquina para sugerir como padrão
SUGGESTED_IP=$(hostname -I | awk '{print $1}')

read -p "$(echo -e ${CYAN}"➡️  Digite o IP desta VPS de Load Balancer [${SUGGESTED_IP}]: "${NC})" -e -i "${SUGGESTED_IP}" LB_IP
read -p "$(echo -e ${CYAN}"➡️  Digite o domínio que aponta para esta VPS (ex: cdn.seudominio.com): "${NC})" LB_DOMAIN
read -p "$(echo -e ${CYAN}"➡️  Digite o domínio do seu painel principal (ex: painel.com): "${NC})" PANEL_DOMAIN
read -p "$(echo -e ${CYAN}"➡️  Digite a Chave Secreta (Secret Key) do seu painel: "${NC})" -e -i "Jean909110" SECRET_KEY
read -p "$(echo -e ${CYAN}"➡️  Digite a Porta de Stream [8080]: "${NC})" -e -i "8080" STREAM_PORT
read -p "$(echo -e ${CYAN}"➡️  Digite a Porta do Agente/Login [80]: "${NC})" -e -i "80" AGENT_PORT

# Versão do PHP instalada no servidor.
PHP_VERSION="8.1"

# --- CONFIRMAÇÃO DOS DADOS ---
clear
echo -e "${GREEN}=====================================================${NC}"
echo -e "${YELLOW}Por favor, confirme se as informações estão corretas:${NC}"
echo -e "${GREEN}=====================================================${NC}"
echo -e "IP do Load Balancer: ${CYAN}${LB_IP}${NC}"
echo -e "Domínio do Load Balancer: ${CYAN}${LB_DOMAIN}${NC}"
echo -e "Domínio do Painel Principal: ${CYAN}${PANEL_DOMAIN}${NC}"
echo -e "Chave Secreta: ${CYAN}${SECRET_KEY}${NC}"
echo -e "Porta de Stream: ${CYAN}${STREAM_PORT}${NC}"
echo -e "Porta do Agente/Login: ${CYAN}${AGENT_PORT}${NC}"
echo -e "${GREEN}=====================================================${NC}"
echo ""

read -p "As informações estão corretas? (s/n): " confirm
if [[ ! $confirm =~ ^[Ss]$ ]]; then
    echo -e "${RED}Instalação cancelada pelo usuário.${NC}"
    exit 1
fi

# --- INÍCIO DA INSTALAÇÃO ---

# PASSO 1: ATUALIZAR O SISTEMA E INSTALAR DEPENDÊNCIAS
echo -e "\n${YELLOW}--> PASSO 1: Atualizando o sistema e instalando pacotes (Nginx, PHP, Curl)...${NC}"
apt update
apt install -y nginx curl php${PHP_VERSION}-fpm

# PASSO 2: CONFIGURAR O FIREWALL (UFW)
echo -e "\n${YELLOW}--> PASSO 2: Configurando o Firewall (UFW)...${NC}"
ufw allow 22/tcp        # Libera a porta SSH (MUITO IMPORTANTE)
ufw allow ${AGENT_PORT}/tcp    # Libera a porta do agente/login
ufw allow ${STREAM_PORT}/tcp   # Libera a porta de streaming
ufw --force enable
echo -e "${GREEN}Firewall ativado e portas liberadas.${NC}"

# PASSO 3: CRIAR O SCRIPT DO AGENTE DE STATUS
echo -e "\n${YELLOW}--> PASSO 3: Criando o agente de status (stats_agent.php)...${NC}"
cat <<EOF > /var/www/html/stats_agent.php
<?php
header('Content-Type: application/json; charset=UTF-8');
\$secret_key = '${SECRET_KEY}';
if (!isset(\$_GET['token']) || \$_GET['token'] !== \$secret_key) {
    http_response_code(403);
    die(json_encode(['status' => 'error', 'message' => 'Acesso Negado']));
}
if (isset(\$_GET['action']) && \$_GET['action'] == 'reboot_services') {
    shell_exec('systemctl restart nginx php${PHP_VERSION}-fpm > /dev/null 2>&1 &');
    echo json_encode(['status' => 'ok', 'message' => 'Comando de reinicialização enviado.']);
    exit;
}
\$uptime_str = trim(shell_exec('uptime -p'));
\$mem_info = shell_exec('free -m');
\$mem_lines = explode("\n", \$mem_info);
\$mem_data = preg_split("/\s+/", \$mem_lines[1]);
\$total_ram = (int)\$mem_data[1];
\$used_ram = (int)\$mem_data[2];
\$ram_usage_percent = (\$total_ram > 0) ? round((\$used_ram / \$total_ram) * 100) : 0;
\$disk_info = shell_exec('df -h / | tail -n 1');
\$disk_data = preg_split("/\s+/", \$disk_info);
\$total_disk = floatval(str_replace(['G', ','], ['', '.'], \$disk_data[1]));
\$used_disk = floatval(str_replace(['G', ','], ['', '.'], \$disk_data[2]));
\$disk_usage_percent = floatval(str_replace('%', '', \$disk_data[4]));
\$response = [
    'status' => 'online', 'uptime' => \$uptime_str,
    'ram' => ['total_mb' => \$total_ram, 'used_mb' => \$used_ram, 'usage_percent' => \$ram_usage_percent],
    'disk' => ['total_gb' => \$total_disk, 'used_gb' => \$used_disk, 'usage_percent' => \$disk_usage_percent],
];
echo json_encode(\$response, JSON_PRETTY_PRINT);
?>
EOF
echo -e "${GREEN}Agente criado com sucesso.${NC}"

# PASSO 4: CRIAR A CONFIGURAÇÃO DO NGINX
echo -e "\n${YELLOW}--> PASSO 4: Criando o arquivo de configuração do Nginx...${NC}"
cat <<EOF > /etc/nginx/sites-available/loadbalancer
server {
    listen ${AGENT_PORT};
    listen [::]:${AGENT_PORT};
    server_name ${LB_IP} ${LB_DOMAIN};
    root /var/www/html;
    index index.php index.html;
    location / { try_files \$uri \$uri/ /index.php?\$query_string; }
    location /player_api.php {
        proxy_pass http://${PANEL_DOMAIN};
        proxy_set_header Host ${PANEL_DOMAIN};
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header User-Agent "NginxProxy";
    }
    location ~ \.php\$ {
        include snippets/fastcgi-php.conf;
        fastcgi_pass unix:/var/run/php/php${PHP_VERSION}-fpm.sock;
    }
}
server {
    listen ${STREAM_PORT};
    listen [::]:${STREAM_PORT};
    server_name ${LB_IP} ${LB_DOMAIN};
    access_log off;
    location / {
        proxy_pass http://${PANEL_DOMAIN};
        proxy_set_header Host ${PANEL_DOMAIN};
        proxy_set_header X-Real-IP \$remote_addr;
        proxy_set_header X-Forwarded-For \$proxy_add_x_forwarded_for;
        proxy_set_header User-Agent "NginxProxy";
    }
}
EOF
echo -e "${GREEN}Arquivo de configuração do Nginx criado.${NC}"

# PASSO 5: ATIVAR O SITE E REINICIAR SERVIÇOS
echo -e "\n${YELLOW}--> PASSO 5: Ativando a configuração e reiniciando os serviços...${NC}"
# Remove o link antigo ou o arquivo padrão para evitar erros
rm -f /etc/nginx/sites-enabled/default
rm -f /etc/nginx/sites-enabled/loadbalancer
# Cria o link simbólico correto
ln -s /etc/nginx/sites-available/loadbalancer /etc/nginx/sites-enabled/loadbalancer
nginx -t
systemctl restart nginx
systemctl restart php${PHP_VERSION}-fpm

echo -e "\n${GREEN}=====================================================${NC}"
echo -e "${GREEN}✅ Instalação do Load Balancer concluída com sucesso!${NC}"
echo -e "${GREEN}=====================================================${NC}"
echo -e "\n${YELLOW}Próximos passos importantes:${NC}"
echo "1. Certifique-se de que o domínio ${LB_DOMAIN} aponta para o IP ${LB_IP}."
echo "2. Adicione este servidor no seu painel de gerenciamento."
echo "3. Teste o login em um aplicativo usando a URL: http://${LB_DOMAIN}:${AGENT_PORT}"