<?php
// =================================================================================
// ARQUIVO DE API DO DASHBOARD (VERSÃO DEFINITIVA, SEGURA E CONFIGURÁVEL)
// =================================================================================

// Força a exibição de todos os erros para facilitar o diagnóstico no futuro
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// Define o cabeçalho para a resposta ser sempre em formato JSON
header('Content-Type: application/json; charset=UTF-8');
session_start();

// --- PASSO 1: VERIFICAÇÃO DE SEGURANÇA (SESSÃO) ---
if (!isset($_SESSION['logged_in_fxtream']) || $_SESSION['logged_in_fxtream'] !== true) {
    http_response_code(403); // Código "Acesso Proibido"
    echo json_encode(['error' => 'Acesso não autorizado. Faça o login no painel.']);
    exit;
}

// --- PASSO 2: VERIFICAÇÃO DE ARQUIVO E CONEXÃO COM O BD ---
$db_path = 'controles/db.php'; // Caminho relativo a este arquivo
if (!file_exists($db_path)) {
    http_response_code(500); // Erro de Servidor
    die(json_encode(['error' => 'ERRO CRÍTICO: O arquivo de conexão com o banco de dados não foi encontrado.']));
}
require_once($db_path);
date_default_timezone_set('America/Sao_Paulo');

try {
    $pdo = conectar_bd();
    if (!$pdo) {
        http_response_code(500);
        die(json_encode(['error' => 'ERRO CRÍTICO: A conexão com o banco de dados falhou. Verifique as credenciais.']));
    }
} catch (Exception $e) {
    http_response_code(500);
    die(json_encode(['error' => 'ERRO CRÍTICO: Exceção ao conectar ao banco de dados: ' . $e->getMessage()]));
}

// --- PASSO 3: LER CONFIGURAÇÃO E LIMPAR CONEXÕES ---
try {
    // ATUALIZADO PARA SEGUNDOS: Lê a configuração em segundos do banco de dados
    $stmt_config = $pdo->prepare("SELECT valor FROM painel_config WHERE chave = 'timeout_conexoes_segundos'");
    $stmt_config->execute();
    $timeout_segundos = $stmt_config->fetchColumn() ?: 10800; // Padrão de 3 horas (10800s) se não encontrar

    // ATUALIZADO PARA SEGUNDOS: Executa a limpeza usando a unidade SECOND
    // CORREÇÃO DE COMPATIBILIDADE: Usa array() para compatibilidade com PHP antigo
    $stmt_delete = $pdo->prepare("DELETE FROM conexoes WHERE ultima_atividade < NOW() - INTERVAL ? SECOND");
    $stmt_delete->execute(array((int)$timeout_segundos));

} catch (Exception $e) {
    http_response_code(500);
    die(json_encode(['error' => 'ERRO CRÍTICO: Falha ao executar a limpeza de conexões: ' . $e->getMessage()]));
}

// --- PASSO 4: BUSCAR E PROCESSAR OS DADOS PARA O DASHBOARD ---
try {
    $loggedInUserId = $_SESSION['admin_id'];
    $userLevel = $_SESSION['nivel_admin'];

    $reseller_filter = '';
    $params = array(); // Usa array() para consistência
    if ($userLevel != 1) {
        $reseller_filter = " AND c.admin_id = :admin_id ";
        $params[':admin_id'] = $loggedInUserId;
    }

    // --- QUERY DETALHADA: Lista todas as conexões ativas ---
    $sql_detalhado = "
        SELECT 
            co.usuario, co.ip, co.ultima_atividade,
            COALESCE(s.name, se.title, 'Conteúdo Desconhecido') AS stream_name,
            IF(s.id IS NOT NULL, s.stream_type, 'series') AS stream_type
        FROM conexoes co
        JOIN clientes c ON co.usuario = c.usuario
        LEFT JOIN streams s ON co.stream_id = s.id
        LEFT JOIN series_episodes se ON co.stream_id = se.id AND s.id IS NULL
        WHERE 1=1 {$reseller_filter}
        ORDER BY co.ultima_atividade DESC;
    ";
    $stmt_detalhado = $pdo->prepare($sql_detalhado);
    $stmt_detalhado->execute($params);
    $conexoes_detalhadas = $stmt_detalhado->fetchAll(PDO::FETCH_ASSOC);

    // --- QUERY AGREGADA: Lista os conteúdos mais assistidos ---
    $sql_agregado = "
        SELECT 
            T.name, T.stream_icon, COUNT(T.conexao_id) as total_espectadores
        FROM (
            SELECT s.name, s.stream_icon, c.id as conexao_id
            FROM conexoes c JOIN clientes cl ON c.usuario = cl.usuario JOIN streams s ON c.stream_id = s.id
            WHERE c.stream_id IS NOT NULL {$reseller_filter}
            UNION ALL
            SELECT se.title as name, se.movie_image as stream_icon, c.id as conexao_id
            FROM conexoes c JOIN clientes cl ON c.usuario = cl.usuario JOIN series_episodes se ON c.stream_id = se.id
            WHERE 1=1 {$reseller_filter}
        ) AS T
        GROUP BY T.name, T.stream_icon ORDER BY total_espectadores DESC, T.name ASC LIMIT 5;
    ";
    $stmt_agregado = $pdo->prepare($sql_agregado);
    $stmt_agregado->execute($params);
    $conteudo_popular = $stmt_agregado->fetchAll(PDO::FETCH_ASSOC);

    // --- PROCESSAMENTO E FORMATAÇÃO DOS DADOS ---
    function formatar_tempo_passado_api($datetime_str) {
        if(!$datetime_str) return 'N/A';
        try {
            $now = new DateTime(); $ago = new DateTime($datetime_str); $diff = $now->diff($ago);
            if ($diff->days > 0) return 'há ' . $diff->days . 'd';
            if ($diff->h > 0) return 'há ' . $diff->h . 'h';
            if ($diff->i > 0) return 'há ' . $diff->i . ' min';
            return 'há ' . $diff->s . ' seg';
        } catch (Exception $e) { return 'N/A'; }
    }
    
    $activity_log = array();
    foreach ($conexoes_detalhadas as $conexao) {
        $activity_log[] = array('usuario' => htmlspecialchars($conexao['usuario']), 'ip' => htmlspecialchars($conexao['ip']), 'assistindo' => htmlspecialchars($conexao['stream_name']), 'ultima_atividade' => formatar_tempo_passado_api($conexao['ultima_atividade']), 'tipo' => htmlspecialchars($conexao['stream_type']));
    }
    $top_content_log = array();
    foreach ($conteudo_popular as $conteudo) {
        $top_content_log[] = array('nome' => htmlspecialchars($conteudo['name']), 'icon' => htmlspecialchars($conteudo['stream_icon']), 'espectadores' => $conteudo['total_espectadores']);
    }
    
    // --- RESPOSTA FINAL EM JSON ---
    $response = array(
        'online_count' => count($conexoes_detalhadas),
        'usuarios_unicos' => count(array_unique(array_column($conexoes_detalhadas, 'usuario'))),
        'activity' => $activity_log,
        'top_channels' => $top_content_log
    );
    
    echo json_encode($response);

} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Erro no servidor ao processar os dados: ' . $e->getMessage()]);
}

$pdo = null;
?>