<?php
/**
 * ARQUIVO: api/api_migracao.php
 * DESCRIÇÃO: Script de back-end para analisar, configurar e importar clientes de listas M3U.
 * CORREÇÃO: Admins com ID 11 (Seu Admin) e Admin Principal (Originalmente 1) são TOTALMENTE ISENTOS de cobrança e verificação de saldo.
 */

// Define que a resposta será sempre em formato JSON
header('Content-Type: application/json');

// Tenta carregar o arquivo de conexão com o banco de dados
try {
    require_once 'controles/db.php';
} catch (Exception $e) {
    echo json_encode(['status' => 'error', 'message' => 'Erro crítico: Não foi possível carregar o arquivo do banco de dados.']);
    exit;
}

// Conecta ao banco de dados
$conexao = conectar_bd();
if (!$conexao) {
    echo json_encode(['status' => 'error', 'message' => 'Falha na conexão com o banco de dados.']);
    exit;
}

// ======================================================================
//  FUNÇÕES DE AJUDA PARA CONFIGURAÇÃO
// ======================================================================

/**
 * Obtém a configuração de cobrança de crédito (1 para cobrar, 0 para não).
 * @return int 1 ou 0.
 */
function get_configuracao_cobranca($conn) {
    try {
        $sql = "SELECT valor FROM configuracoes WHERE chave = 'migracao_cobrar_credito'";
        $stmt = $conn->query($sql);
        $resultado = $stmt->fetchColumn();
        // Assume cobrar se não houver configuração
        return $resultado !== false ? (int)$resultado : 1;
    } catch (PDOException $e) {
        return 1;
    }
}

/**
 * Salva a configuração (UPDATE/INSERT robusto).
 * @return bool
 */
function set_configuracao_cobranca($conn, $valor) {
    $chave = 'migracao_cobrar_credito';
    $valor_str = (string)$valor; 

    try {
        // 1. Tenta atualizar
        $sql_update = "UPDATE configuracoes SET valor = :valor WHERE chave = :chave";
        $stmt_update = $conn->prepare($sql_update);
        $stmt_update->bindParam(':valor', $valor_str, PDO::PARAM_STR); 
        $stmt_update->bindParam(':chave', $chave, PDO::PARAM_STR);
        $stmt_update->execute();

        if ($stmt_update->rowCount() > 0) {
            return true;
        }

        // 2. Se não atualizou, insere
        $sql_insert = "INSERT INTO configuracoes (chave, valor) VALUES (:chave, :valor)";
        $stmt_insert = $conn->prepare($sql_insert);
        $stmt_insert->bindParam(':chave', $chave, PDO::PARAM_STR);
        $stmt_insert->bindParam(':valor', $valor_str, PDO::PARAM_STR);
        
        return $stmt_insert->execute();

    } catch (PDOException $e) {
        return false;
    }
}


// ======================================================================
//  VERIFICAÇÃO DE SEGURANÇA (TOKEN DIRETO NO BANCO)
// ======================================================================
$token_enviado = $_POST['token'] ?? '';

if (empty($token_enviado)) {
    echo json_encode(['status' => 'error', 'message' => 'Erro de autenticação: Token não fornecido.']);
    exit;
}

// Valida o token no banco de dados e obtém o ID do admin
try {
    $sql_token = "SELECT id FROM admin WHERE token = :token";
    $stmt_token = $conexao->prepare($sql_token);
    $stmt_token->bindParam(':token', $token_enviado, PDO::PARAM_STR);
    $stmt_token->execute();
    $admin_logado = $stmt_token->fetch(PDO::FETCH_ASSOC);

    if (!$admin_logado || empty($admin_logado['id'])) {
        echo json_encode(['status' => 'error', 'message' => 'Erro de autenticação: Token inválido ou expirado. Faça login novamente.']);
        exit;
    }
    $admin_id_logado = $admin_logado['id'];
} catch (PDOException $e) {
    echo json_encode(['status' => 'error', 'message' => 'Erro no banco de dados durante a validação do token.']);
    exit;
}
// ======================================================================

// Define o custo por dia para a importação
define('CUSTO_POR_DIA', 0.05);  
// **AJUSTE AQUI:** Define o ID do Admin que será ISENTO de cobrança.
// MUDANÇA: O ID 11 agora é o Admin Principal para fins de isenção nesta API.
define('ADMIN_PRINCIPAL_ID', 11); // <-- CORRIGIDO PARA ISENTAR O ID 11!

// Roteador de ações
$action = $_POST['action'] ?? '';

if ($action === 'analisar') {
    analisarLink($conexao, $admin_id_logado);
} elseif ($action === 'importar') {
    importarCliente($conexao, $admin_id_logado);
} elseif ($action === 'salvar_config') {
    salvarConfiguracao($conexao);
} else {
    echo json_encode(['status' => 'error', 'message' => 'Ação inválida solicitada.']);
}

// ======================================================================
//  FUNÇÃO DE SALVAR CONFIGURAÇÃO
// ======================================================================

/**
 * Salva a nova configuração de cobrança de créditos.
 */
function salvarConfiguracao($conn) {
    // Verifique se o admin_id_logado está disponível globalmente ou passe-o como parâmetro,
    // para esta função funcionar corretamente, vamos confiar no fluxo anterior que definiu $admin_id_logado
    global $admin_id_logado; 
    
    // Admins Principais não devem ter a opção de desligar a cobrança para Revendedores
    // Mas, como essa tela é crucial para Admins, vamos permitir que ele configure,
    // mas garantindo que ele não seja cobrado.
    
    $cobrar = (int)($_POST['cobrar_credito'] ?? 1); 

    if (set_configuracao_cobranca($conn, $cobrar)) {
        echo json_encode(['status' => 'success', 'message' => 'Configuração de cobrança salva com sucesso!']);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Falha ao salvar a configuração no banco de dados.']);
    }
    exit;
}


// ======================================================================
//  FUNÇÕES DE ANÁLISE E IMPORTAÇÃO
// ======================================================================

/**
 * Analisa o link M3U, conecta-se ao servidor de origem e retorna os dados.
 */
function analisarLink($conn, $admin_id_logado) {
    $m3u_url = $_POST['url'] ?? '';

    if (empty($m3u_url) || !filter_var($m3u_url, FILTER_VALIDATE_URL)) {
        echo json_encode(['status' => 'error', 'message' => 'URL M3U inválida ou em branco.']);
        return;
    }

    $parts = parse_url($m3u_url);
    if (!isset($parts['host']) || !isset($parts['query'])) {
        echo json_encode(['status' => 'error', 'message' => 'O formato da URL não parece ser de uma lista M3U válida.']);
        return;
    }

    $servidor = $parts['scheme'] . '://' . $parts['host'] . (isset($parts['port']) ? ':' . $parts['port'] : '');
    parse_str($parts['query'], $query);
    $usuario = $query['username'] ?? '';
    $senha = $query['password'] ?? '';

    if (empty($usuario) || empty($senha)) {
        echo json_encode(['status' => 'error', 'message' => 'Não foi possível encontrar usuário e senha na URL.']);
        return;
    }

    $api_url = "{$servidor}/player_api.php?username={$usuario}&password={$senha}";
    
    // Lógica de cURL (Mantida)
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $api_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
    curl_setopt($ch, CURLOPT_TIMEOUT, 15);
    $response = curl_exec($ch);
    $httpcode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    if($httpcode != 200){
        echo json_encode(['status' => 'error', 'message' => 'O servidor de origem não respondeu (Código: '.$httpcode.'). Pode estar offline ou bloqueando a conexão.']);
        return;
    }

    $data = json_decode($response, true);

    if (isset($data['user_info']['auth']) && $data['user_info']['auth'] == 1) {
        $info = $data['user_info'];
        
        $dias_restantes = 0;
        if (!empty($info['exp_date'])) {
            $exp_timestamp = (int)$info['exp_date'];
            $hoje_timestamp = time();
            if ($exp_timestamp > $hoje_timestamp) {
                $dias_restantes = floor(($exp_timestamp - $hoje_timestamp) / (60 * 60 * 24));
            }
        }
        
        // >>> LÓGICA DE COBRANÇA PARA EXIBIÇÃO NO FRONTEND <<<
        $cobrar_credito = get_configuracao_cobranca($conn);
        
        // Admins Principais (ID 11) NUNCA SÃO COBRADOS (Apenas para exibição no frontend)
        if ((int)$admin_id_logado === ADMIN_PRINCIPAL_ID) {
            $custo = 0.00;
            $cobrar_credito = 0; // Sobrescreve para o frontend exibir 'GRATUITO'
        } else {
            // Custo real para revendedores
            $custo = $cobrar_credito ? $dias_restantes * CUSTO_POR_DIA : 0.00;
        }

        $resultado = [
            'servidor' => $servidor,
            'usuario' => $usuario,
            'senha' => $senha,
            'conexoes' => $info['max_connections'] ?? '1',
            'dias_restantes' => $dias_restantes,
            'custo_importacao' => number_format($custo, 2, ',', '.'),
            'exp_date_timestamp' => $info['exp_date'],
            'cobrar_credito' => $cobrar_credito // Flag para o JS (0 = Gratuito, 1 = Cobrar)
        ];

        echo json_encode(['status' => 'success', 'data' => $resultado]);

    } else {
        echo json_encode(['status' => 'error', 'message' => 'Credenciais inválidas ou usuário inativo no servidor de origem.']);
    }
}

/**
 * Importa o cliente para o banco de dados local, DÉBITANDO o crédito do revendedor.
 */
function importarCliente($conn, $admin_id_autenticado) {
    
    // ==================================================
    //               INÍCIO DA CORREÇÃO
    // ==================================================
    // 1. Recebe o TEXTO JSON
    $cliente_json = $_POST['cliente'] ?? null; 
    if (!$cliente_json) {
        echo json_encode(['status' => 'error', 'message' => 'Dados do cliente não foram recebidos para importação.']);
        return;
    }

    // 2. Transforma o TEXTO de volta em um array PHP
    $cliente = json_decode($cliente_json, true); 

    // 3. Verifica se a "tradução" funcionou
    if (!$cliente) {
        echo json_encode(['status' => 'error', 'message' => 'Erro ao decodificar dados do cliente (JSON inválido).']);
        return;
    }
    // ==================================================
    //                FIM DA CORREÇÃO
    // ==================================================


    // --- DADOS DO CLIENTE ---
    $plano_id_padrao = 55;
    $usuario = $cliente['usuario'];
    $senha = $cliente['senha'];
    $conexoes = (int)$cliente['conexoes'];
    $exp_date = (int)$cliente['exp_date_timestamp'];
    $vencimento_formatado = date('Y-m-d H:i:s', $exp_date);
    
    $dias_restantes = (int)$cliente['dias_restantes'];
    $custo_real = $dias_restantes * CUSTO_POR_DIA;
    
    // >>> LÓGICA DE COBRANÇA E DÉBITO (CORRIGIDA) <<<
    $cobrar_credito = get_configuracao_cobranca($conn);
    $custo_a_debitar = 0.00; // Começa em zero
    
    // Condição para COBRAR: É revendedor E a cobrança está ativada no painel
    // Se o admin logado FOR o ADMIN_PRINCIPAL_ID (agora 11), a condição é FALSA e o custo_a_debitar permanece 0.
    $is_revendedor_e_cobrar = ((int)$admin_id_autenticado !== ADMIN_PRINCIPAL_ID && $cobrar_credito == 1);

    if ($is_revendedor_e_cobrar) {
        $custo_a_debitar = $custo_real;
    }


    // SE FOR REVENDEDOR E HOUVER CUSTO, VERIFICA O SALDO E DEBITA
    if ($custo_a_debitar > 0) {
        
        $conn->beginTransaction(); 
        try {
            // 1. OBTÉM O SALDO DO REVENDEDOR
            $sql_saldo = "SELECT creditos_usados FROM admin WHERE id = :admin_id FOR UPDATE";
            $stmt_saldo = $conn->prepare($sql_saldo);
            $stmt_saldo->bindParam(':admin_id', $admin_id_autenticado, PDO::PARAM_INT);
            $stmt_saldo->execute();
            
            // Seu saldo é armazenado como NEGATIVO na coluna 'creditos_usados', então invertemos o sinal
            $creditos_usados = (float)$stmt_saldo->fetchColumn();
            $saldo_atual = -$creditos_usados; 

            // 2. VERIFICA SE HÁ CRÉDITO SUFICIENTE
            if ($saldo_atual < $custo_a_debitar) {
                $conn->rollBack(); 
                echo json_encode(['status' => 'error', 'message' => 'Saldo insuficiente! Custo da migração: R$' . number_format($custo_a_debitar, 2, ',', '.') . '. Saldo atual: R$' . number_format($saldo_atual, 2, ',', '.')]);
                return; // INTERROMPE TUDO AQUI PARA REVENDEDORES SEM SALDO
            }

            // 3. DÉBITO DO CRÉDITO 
            $novo_credito_usado = $creditos_usados + $custo_a_debitar;
            $sql_debito = "UPDATE admin SET creditos_usados = :novo_credito_usado WHERE id = :admin_id";
            $stmt_debito = $conn->prepare($sql_debito);
            $stmt_debito->bindParam(':novo_credito_usado', $novo_credito_usado);
            $stmt_debito->bindParam(':admin_id', $admin_id_autenticado, PDO::PARAM_INT);
            $stmt_debito->execute();

        } catch (PDOException $e) {
            $conn->rollBack();
            echo json_encode(['status' => 'error', 'message' => 'Erro de banco de dados durante o débito de créditos.']);
            return;
        }
    }
    // ======================================================================
    // AQUI, O ADMIN PRINCIPAL (ID 11) SEGUE, POIS O 'if ($custo_a_debitar > 0)' FOI IGNORADO PARA ELE.

    // INSERÇÃO DO CLIENTE NO BANCO DE DADOS
    if (!$conn->inTransaction()) {
        $conn->beginTransaction();
    }
    
    try {
        $sql = "INSERT INTO clientes (name, usuario, senha, plano, criado_em, Vencimento, is_trial, adulto, conexoes, bloqueio_conexao, admin_id) 
                VALUES (?, ?, ?, ?, NOW(), ?, ?, ?, ?, ?, ?)";
        
        $stmt = $conn->prepare($sql);
        
        $stmt->execute([
            $usuario,
            $usuario,
            $senha,
            $plano_id_padrao,
            $vencimento_formatado,
            0, // is_trial
            0, // adulto
            $conexoes,
            'nao', // bloqueio_conexao
            $admin_id_autenticado
        ]);

        $conn->commit(); // Finaliza a transação (Débito, se houver, + Inserção)
        
        // Mensagem dinâmica baseada se houve débito ou se é o admin principal
        $mensagem = ((int)$admin_id_autenticado === ADMIN_PRINCIPAL_ID)
                    ? 'Cliente importado com sucesso! Migração gratuita para o Admin (ID ' . ADMIN_PRINCIPAL_ID . ').'
                    : 'Cliente importado com sucesso! Foi debitado R$' . number_format($custo_a_debitar, 2, ',', '.') . ' em créditos.';

        echo json_encode(['status' => 'success', 'message' => $mensagem]);
        
    } catch (PDOException $e) {
        // Se a transação foi iniciada (no caso de revendedores), faz rollback.
        if ($conn->inTransaction()) {
            $conn->rollBack(); 
        }
        
        if ($e->getCode() == 23000) {
             echo json_encode(['status' => 'error', 'message' => 'Erro: Este nome de usuário já existe no seu sistema.']);
        } else {
             echo json_encode(['status' => 'error', 'message' => 'Erro de banco de dados: ' . $e->getMessage()]);
        }
    }
}
?>