<?php
/**
 * ARQUIVO: /gerenciador/ajax/temporada_excluir.php
 * DESCRIÇÃO: Script para exclusão de uma única temporada e seus episódios.
 */

// Define que a resposta será no formato JSON, que o JavaScript entende.
header('Content-Type: application/json');

// Inclui a conexão com o banco de dados.
// Usamos ../../ porque estamos duas pastas "acima" do diretório raiz da API.
// (Saindo de /ajax/ e depois de /gerenciador/)
require_once('../../api/controles/db.php');

// Função para padronizar a resposta
function responder($sucesso, $mensagem) {
    echo json_encode(['success' => $sucesso, 'message' => $mensagem]);
    exit(); // Encerra o script após enviar a resposta
}

// Verifica se a requisição é do tipo POST (mais seguro)
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    responder(false, 'Acesso negado.');
}

// Pega e valida o ID da temporada que veio do formulário
$id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
if (!$id) {
    responder(false, 'ID da temporada inválido.');
}

try {
    $pdo = conectar_bd();
    
    // Inicia uma transação: ou tudo funciona, ou nada é alterado.
    $pdo->beginTransaction();

    // Passo 1: Excluir todos os episódios que pertencem a esta temporada.
    // CORREÇÃO APLICADA: Alterado "season_id" para "season" para corresponder à sua tabela.
    $stmt_episodios = $pdo->prepare("DELETE FROM series_episodes WHERE season = ?");
    $stmt_episodios->execute([$id]);

    // Passo 2: Agora que os episódios foram removidos, exclui a temporada.
    $stmt_temporada = $pdo->prepare("DELETE FROM series_seasons WHERE id = ?");
    $stmt_temporada->execute([$id]);

    // Se chegou até aqui sem erros, confirma as exclusões no banco de dados.
    $pdo->commit();

    responder(true, 'Temporada e todos os seus episódios foram excluídos com sucesso!');

} catch (PDOException $e) {
    // Se ocorreu qualquer erro, desfaz todas as operações da transação.
    if (isset($pdo) && $pdo->inTransaction()) {
        $pdo->rollBack();
    }
    
    // Em um site real, você registraria o erro em um log em vez de mostrá-lo.
    // error_log($e->getMessage()); 
    responder(false, 'Erro no banco de dados ao tentar excluir a temporada.');
}