<?php
// ARQUIVO: /gerenciador/conexoes_online.php
// (VERSÃO INTEGRADA AO PAINEL)

// Força a exibição de todos os erros do PHP.
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

// VERIFICAÇÃO 1: O ARQUIVO DE CONEXÃO EXISTE?
// Caminho corrigido para subir um nível (de 'gerenciador' para a raiz)
$db_path = '../api/controles/db.php'; 
if (!file_exists($db_path)) {
    die("ERRO CRÍTICO: O arquivo de conexão com o banco de dados não foi encontrado. Verifique se o caminho está correto. Caminho esperado: [ " . realpath(dirname(__FILE__)) . "/" . $db_path . " ]");
}
require_once($db_path);

// ==================================================================
// ### CORREÇÃO ###
// O Bloco session_start() foi REMOVIDO daqui,
// pois o menu.php (linha 2) já inicia a sessão.
// ==================================================================


// ==================================================================
// ### INÍCIO DA INTEGRAÇÃO ###
// 1. Inclui o seu menu (caminho corrigido)
// O menu.php JÁ FAZ A VERIFICAÇÃO DE LOGIN
// ==================================================================
require_once("../menu.php");
// ==================================================================

date_default_timezone_set('America/Sao_Paulo');

try {
    $pdo = conectar_bd();
    if (!$pdo) {
        die("ERRO CRÍTICO: A conexão com o banco de dados falhou. A função conectar_bd() não retornou um objeto válido. Verifique as credenciais no arquivo db.php.");
    }
} catch (Exception $e) {
    die("ERRO CRÍTICO: Uma exceção ocorreu ao tentar conectar ao banco de dados: " . $e->getMessage());
}

try {
    // Lendo a configuração de timeout do banco de dados
    $stmt_config = $pdo->prepare("SELECT valor FROM painel_config WHERE chave = 'timeout_conexoes_minutos'");
    $stmt_config->execute();
    $timeout_minutos = $stmt_config->fetchColumn() ?: 180;

    // LIMPEZA AUTOMÁTICA DE CONEXÕES INATIVAS
    $stmt_delete = $pdo->prepare("DELETE FROM conexoes WHERE ultima_atividade < NOW() - INTERVAL ? MINUTE");
    $stmt_delete->execute(array((int)$timeout_minutos));

} catch (Exception $e) {
    // Falha na limpeza de conexões inativas (apenas loga/ignora)
}

// --- CONSULTAS DE DADOS ---
// Assumindo que o menu.php define 'admin_id' e 'nivel_admin' na sessão
$loggedInUserId = $_SESSION['admin_id'];
$userLevel = $_SESSION['nivel_admin'];
$reseller_filter_sql = '';
if ($userLevel != 1) {
    // Filtro para revendedores: só vê conexões de seus clientes
    $reseller_filter_sql = " AND c.admin_id = :admin_id ";
}

// 1. Canais Mais Populares (Lives)
$sql_agregado = "
    SELECT 
        s.name, s.stream_icon, COUNT(co.id) AS total_espectadores 
    FROM conexoes co 
    JOIN clientes c ON co.usuario = c.usuario 
    JOIN streams s ON co.stream_id = s.id 
    WHERE co.stream_id IS NOT NULL AND s.stream_type = 'live' {$reseller_filter_sql} 
    GROUP BY co.stream_id 
    ORDER BY total_espectadores DESC, s.name ASC 
    LIMIT 10;";
$stmt_agregado = $pdo->prepare($sql_agregado);
if ($userLevel != 1) {
    $stmt_agregado->bindValue(':admin_id', $loggedInUserId, PDO::PARAM_INT);
}
$stmt_agregado->execute();
$canais_populares = $stmt_agregado->fetchAll(PDO::FETCH_ASSOC);

// 2. Conexões Detalhadas (QUERY CORRIGIDA)
$sql_detalhado = "
    SELECT 
        co.usuario, co.ip, co.user_agent, co.ultima_atividade,
        IFNULL(s.name, se.title) AS stream_name,
        IFNULL(s.stream_type, 'series') AS stream_type -- Mantém a dedução do tipo aqui
    FROM conexoes co
    JOIN clientes c ON co.usuario = c.usuario
    LEFT JOIN streams s ON co.stream_id = s.id
    LEFT JOIN series_episodes se ON s.id IS NULL AND co.stream_id = se.id
    WHERE 1=1 {$reseller_filter_sql}
    ORDER BY co.ultima_atividade DESC;";
$stmt_detalhado = $pdo->prepare($sql_detalhado);
if ($userLevel != 1) {
    $stmt_detalhado->bindValue(':admin_id', $loggedInUserId, PDO::PARAM_INT);
}
$stmt_detalhado->execute();
$conexoes_detalhadas = $stmt_detalhado->fetchAll(PDO::FETCH_ASSOC);

// 3. Estatísticas
$total_conexoes = count($conexoes_detalhadas);
$usuarios_unicos = count(array_unique(array_column($conexoes_detalhadas, 'usuario')));
$streams_unicos_online = count(array_filter(array_unique(array_column($conexoes_detalhadas, 'stream_name'))));


// Função de formatação de tempo
function formatar_tempo_passado($datetime_str) {
    if (empty($datetime_str)) return 'N/A';
    try {
        $agora = new DateTime(); 
        $ultima_atividade = new DateTime($datetime_str);
        $diff = $agora->diff($ultima_atividade);
        
        if ($diff->days > 0) return 'há ' . $diff->days . 'd';
        if ($diff->h > 0) return 'há ' . $diff->h . 'h';
        if ($diff->i > 0) return 'há ' . $diff->i . ' min';
        return 'há ' . $diff->s . ' seg';
    } catch (Exception $e) { 
        return 'Data inválida'; 
    }
}
?>

<style>
    .stat-card { background: linear-gradient(135deg, #36469E 0%, #4D3269 100%); color:white; }
    
    .card { 
        background-color: var(--bs-card-bg); 
        border: 1px solid var(--bs-border-color); 
    }
    
    .table tbody td,
    .table tbody td strong {
        color: var(--bs-body-color) !important; 
        vertical-align: middle;
    }
    .table tbody td strong {
        font-weight: 600;
        color: var(--bs-heading-color) !important; 
    }

    .list-group-item {
        background-color: var(--bs-card-bg) !important; 
        color: var(--bs-body-color) !important; 
        border-color: var(--bs-border-color) !important;
    }
</style>

<div class="container-fluid p-md-4 p-2">
    
    <div class="d-flex justify-content-between align-items-center mb-4">
        <h1 class="h3 mb-0"><i class="fas fa-users-cog me-2 text-primary"></i>Monitor de Usuários Online</h1>
    </div>
    
    <div class="row mb-4 g-3">
        <div class="col-md-3">
            <div class="card stat-card p-3 h-100">
                <div class="d-flex align-items-center">
                    <i class="fas fa-globe-americas fa-3x me-3 opacity-75"></i>
                    <div> <h5 class="mb-0"><?= $total_conexoes ?></h5> <small>Conexões Ativas</small> </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card stat-card p-3 h-100">
                <div class="d-flex align-items-center">
                    <i class="fas fa-users fa-3x me-3 opacity-75"></i>
                    <div> <h5 class="mb-0"><?= $usuarios_unicos ?></h5> <small>Clientes Únicos Online</small> </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card stat-card p-3 h-100">
                <div class="d-flex align-items-center">
                    <i class="fas fa-tv fa-3x me-3 opacity-75"></i>
                    <div> <h5 class="mb-0"><?= $streams_unicos_online ?></h5> <small>Streams Sendo Vistos</small> </div>
                </div>
            </div>
        </div>
        <div class="col-md-3">
            <div class="card stat-card p-3 h-100" style="background: linear-gradient(135deg, #8a0101 0%, #d42626 100%);">
                <div class="d-flex align-items-center justify-content-between">
                    <i class="fas fa-power-off fa-3x me-3 opacity-75"></i>
                    <div class="text-end"> 
                        <h5 class="mb-1">Ações de Controle</h5> 
                        <button id="kill-all-btn" class="btn btn-sm btn-light mt-1" title="Finaliza todas as conexões visíveis na tabela."><i class="fas fa-times-circle me-1"></i> Matar Todas</button>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <div class="row g-4">
        <div class="col-lg-5">
            <div class="card h-100">
                <div class="card-header"><strong><i class="fas fa-fire me-2 text-danger"></i>Canais Mais Assistidos (Em Live)</strong></div>
                <div class="card-body p-0">
                    <ul class="list-group list-group-flush">
                        <?php if (empty($canais_populares)): ?>
                            <li class="list-group-item">Nenhum canal ao vivo sendo assistido.</li>
                        <?php else: foreach ($canais_populares as $canal): ?>
                            <li class="list-group-item d-flex justify-content-between align-items-center">
                                <div>
                                    <img src="<?= htmlspecialchars($canal['stream_icon']) ?>" style="width:24px; height:24px; margin-right:10px; border-radius:4px; background-color: #333;" alt="" onerror="this.style.display='none'">
                                    <?= htmlspecialchars($canal['name']) ?>
                                </div>
                                <span class="badge bg-danger rounded-pill"><?= $canal['total_espectadores'] ?></span>
                            </li>
                        <?php endforeach; endif; ?>
                    </ul>
                </div>
            </div>
        </div>
        <div class="col-lg-7">
            <div class="card h-100">
                <div class="card-header"><strong><i class="fas fa-list-ul me-2 text-primary"></i>Log de Conexões Ativas</strong></div>
                <div class="card-body p-0" style="max-height: 600px; overflow-y: auto;">
                    <div class="table-responsive">
                        <table class="table table-sm table-hover mb-0"> 
                            <thead style="position: sticky; top: 0; z-index: 1;">
                                <tr>
                                    <th>Usuário</th>
                                    <th>Assistindo</th>
                                    <th>IP</th>
                                    <th>Última Atividade</th>
                                    <th class="text-center">Ação</th> </tr>
                            </thead>
                            <tbody>
                                <?php if (empty($conexoes_detalhadas)): ?>
                                    <tr><td colspan="5" class="text-center p-3">Nenhuma conexão ativa no momento.</td></tr>
                                <?php else: foreach ($conexoes_detalhadas as $conexao): ?>
                                    <tr>
                                        <td><strong><?= htmlspecialchars($conexao['usuario']) ?></strong></td>
                                        <td>
                                            <?php
                                                $tipo = htmlspecialchars($conexao['stream_type'] ?? 'vod');
                                                $cor_badge = 'secondary';
                                                if ($tipo === 'live') $cor_badge = 'success';
                                                if ($tipo === 'movie') $cor_badge = 'info';
                                            ?>
                                            <span class="badge bg-<?= $cor_badge ?> me-2"><?= ucfirst($tipo) ?></span>
                                            <?= htmlspecialchars($conexao['stream_name'] ?? 'Conteúdo VOD') ?>
                                        </td>
                                        <td><?= htmlspecialchars($conexao['ip']) ?></td>
                                        <td><small class="text-muted"><?= formatar_tempo_passado($conexao['ultima_atividade']) ?></small></td>
                                        <td class="text-center">
                                            <button class="btn btn-danger btn-sm kill-individual-btn" 
                                                    data-usuario="<?= htmlspecialchars($conexao['usuario']) ?>" 
                                                    data-ip="<?= htmlspecialchars($conexao['ip']) ?>" 
                                                    title="Finalizar Conexão">
                                                <i class="fas fa-times"></i>
                                            </button>
                                        </td>
                                    </tr>
                                <?php endforeach; endif; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div> <script>
    document.addEventListener('DOMContentLoaded', function() {
        // O endpoint é 'acoes_conexoes.php'
        const endpoint = 'acoes_conexoes.php'; 

        function killConnectionAction(action, data = {}) {
            Swal.fire({
                title: action === 'kill_all' ? 'Tem certeza?' : 'Finalizar Conexão?',
                text: action === 'kill_all' ? "Isso irá desconectar TODAS as conexões ativas que você tem permissão para gerenciar!" : `Deseja forçar o fechamento da conexão do usuário **${data.usuario}** (IP: ${data.ip})?`,
                icon: 'warning',
                showCancelButton: true,
                confirmButtonColor: '#dc3545',
                cancelButtonColor: '#6c757d',
                confirmButtonText: action === 'kill_all' ? 'Sim, Matar Todas!' : 'Sim, Finalizar!',
                cancelButtonText: 'Cancelar',
                allowOutsideClick: false,
                allowHtml: true 
            }).then((result) => {
                if (result.isConfirmed) {
                    Swal.fire({
                        title: 'Processando...',
                        text: 'Aguarde um momento. A página será recarregada.',
                        allowOutsideClick: false,
                        didOpen: () => {
                            Swal.showLoading();
                        }
                    });

                    const formData = new FormData();
                    formData.append('action', action);
                    for (const key in data) {
                        formData.append(key, data[key]);
                    }

                    fetch(endpoint, {
                        method: 'POST',
                        body: formData
                    })
                    .then(response => {
                        if (!response.ok) {
                            return response.text().then(text => {
                                throw new Error(`Erro ${response.status}: ${text.substring(0, 100)}...`);
                            });
                        }
                        return response.json();
                    })
                    .then(result => {
                        if (result.status === 'success' || result.status === 'warning') {
                            Swal.fire({
                                icon: result.status,
                                title: 'Ação Concluída!',
                                text: result.message,
                                timer: 3000,
                                timerProgressBar: true,
                                allowHtml: true 
                            }).then(() => {
                                window.location.reload(); 
                            });
                        } else {
                            Swal.fire('Erro!', result.message, 'error');
                        }
                    })
                    .catch(error => {
                        console.error('Erro na requisição:', error);
                        Swal.fire('Erro Crítico!', 'Não foi possível completar a ação. Verifique o console ou o arquivo **acoes_conexoes.php**.' + (error.message ? `<br><br>Detalhe: ${error.message}` : ''), 'error');
                    });
                }
            });
        }

        const killAllBtn = document.getElementById('kill-all-btn');
        if(killAllBtn) {
            killAllBtn.addEventListener('click', function() {
                killConnectionAction('kill_all');
            });
        }

        document.querySelectorAll('.kill-individual-btn').forEach(button => {
            button.addEventListener('click', function() {
                const usuario = this.dataset.usuario;
                const ip = this.dataset.ip;
                killConnectionAction('kill_individual', { usuario: usuario, ip: ip });
            });
        });
    });
</script>

<?php
// ==================================================================
// ### INÍCIO DA INTEGRAÇÃO ###
// 2. Inclui o seu footer (caminho corrigido)
// ==================================================================
require_once("../footer.php"); 
?>