<?php
// Arquivo: /gerenciador_db.php - VERSÃO FINAL CORRIGIDA
// Adicionado fallback para backup em Puro PHP caso exec() esteja desabilitado.
// [CORREÇÃO MEMÓRIA] Modificado backup_database_php para usar fetch() (linha-a-linha) em vez de fetchAll().

require_once(__DIR__ . '/api/controles/db.php');

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
if (empty($_SESSION['admin_id'])) {
    header('Location: ./index.php');
    exit();
}

// --- CONFIGURAÇÕES ---
$backup_dir = __DIR__ . '/backups/'; // Caminho absoluto para a pasta de backups
$feedback = '';

// --- DEFINIÇÃO DAS CONSTANTES DE CONEXÃO ---
// O seu arquivo 'api/controles/db.php' usa uma função 'conectar_bd()'
// mas este script (mysqldump) precisa de constantes globais.
// Estamos definindo-as aqui com base nos valores do seu db.php.
// NOTA: O ideal seria mover estas 4 linhas PARA o seu 'api/controles/db.php'.
if (!defined('DB_HOST')) define('DB_HOST', 'localhost');
if (!defined('DB_USER')) define('DB_USER', 'user_officextream');
if (!defined('DB_PASS')) define('DB_PASS', 'pass_officextream');
if (!defined('DB_NAME')) define('DB_NAME', 'officextream');


// --- FUNÇÃO DE BACKUP EM PURO PHP ---
/**
 * Cria um backup do banco de dados usando PDO (Puro PHP)
 * Usado como fallback se 'exec' (mysqldump) estiver desabilitado.
 * * @param PDO $pdo Objeto de conexão PDO
 * @param string $output_file Caminho completo do arquivo de saída .sql
 * @return bool|string Retorna true em sucesso, ou string de erro em falha.
 */
function backup_database_php($pdo, $output_file) {
    try {
        $handle = fopen($output_file, 'w');
        if ($handle === false) {
            return "Não foi possível abrir o arquivo de saída para escrita: " . $output_file;
        }

        fwrite($handle, "-- Backup do banco de dados " . DB_NAME . " gerado em " . date('Y-m-d H:i:s') . "\n");
        fwrite($handle, "-- Gerado por script PHP (fallback de exec())\n\n");
        fwrite($handle, "SET NAMES 'utf8mb4';\n");
        fwrite($handle, "SET FOREIGN_KEY_CHECKS=0;\n");
        fwrite($handle, "SET SQL_MODE='NO_AUTO_VALUE_ON_ZERO';\n");
        fwrite($handle, "START TRANSACTION;\n\n");

        $tables_stmt = $pdo->query('SHOW TABLES');
        $tables = $tables_stmt->fetchAll(PDO::FETCH_NUM);

        foreach ($tables as $table_row) {
            $table = $table_row[0];

            // 1. Obter e escrever a estrutura da tabela (CREATE TABLE)
            $create_table_stmt = $pdo->query("SHOW CREATE TABLE `${table}`");
            $create_table_row = $create_table_stmt->fetch(PDO::FETCH_ASSOC);
            $create_table_stmt->closeCursor();
            
            fwrite($handle, "\n-- --------------------------------------------------------\n");
            fwrite($handle, "-- Estrutura da tabela `${table}`\n");
            fwrite($handle, "-- --------------------------------------------------------\n\n");
            fwrite($handle, $create_table_row['Create Table'] . ";\n\n");

            // 2. Obter e escrever os dados (INSERT INTO)
            fwrite($handle, "-- --------------------------------------------------------\n");
            fwrite($handle, "-- Despejando dados para a tabela `${table}`\n");
            fwrite($handle, "-- --------------------------------------------------------\n\n");

            $data_stmt = $pdo->query("SELECT * FROM `${table}`");
            $row_count = $data_stmt->rowCount();

            // --- [INÍCIO DA CORREÇÃO DE MEMÓRIA] ---
            if ($row_count > 0) {
                // Flags para processamento linha-a-linha
                $is_first_row = true;
                $field_names = [];
                $total_rows_processed = 0;

                // Alterado: Usar while fetch() (linha-a-linha) em vez de fetchAll() (tudo de uma vez)
                while ($row = $data_stmt->fetch(PDO::FETCH_ASSOC)) {
                    
                    // Na primeira linha, pegamos os nomes das colunas e escrevemos o cabeçalho do INSERT
                    if ($is_first_row) {
                        $field_names = array_keys($row);
                        $insert_prefix = "INSERT INTO `${table}` (`" . implode('`, `', $field_names) . "`) VALUES \n";
                        fwrite($handle, $insert_prefix);
                        $is_first_row = false; // Desativa o flag
                    } else {
                        // Se não for a primeira linha, escreve a vírgula da linha anterior
                        fwrite($handle, ",\n");
                    }

                    // Processa os valores (código que já existia, mas movido para dentro do while)
                    $values = [];
                    foreach ($row as $value) { // Usa a $row atual do while
                        if ($value === null) {
                            $values[] = 'NULL';
                        } else {
                            // Escapa o valor
                            $values[] = $pdo->quote($value);
                        }
                    }
                    fwrite($handle, "(" . implode(', ', $values) . ")");
                    
                    $total_rows_processed++;
                }
                
                // Se processamos alguma linha, temos que fechar o comando INSERT
                if ($total_rows_processed > 0) {
                     fwrite($handle, ";\n"); // Fim do statement
                }

            } else {
                 fwrite($handle, "-- Tabela `${table}` está vazia.\n");
            }
            // --- [FIM DA CORREÇÃO DE MEMÓRIA] ---

            $data_stmt->closeCursor();
            fwrite($handle, "\n");
        }

        fwrite($handle, "\nCOMMIT;\n");
        fwrite($handle, "SET FOREIGN_KEY_CHECKS=1;\n");
        fclose($handle);
        return true;

    } catch (PDOException $e) {
        if (isset($handle) && $handle !== false) fclose($handle);
        if (file_exists($output_file)) @unlink($output_file);
        return "Erro de PDO: " . $e->getMessage();
    } catch (Exception $e) {
        if (isset($handle) && $handle !== false) fclose($handle);
        if (file_exists($output_file)) @unlink($output_file);
        return "Erro geral: " . $e->getMessage();
    }
}


// --- VERIFICAÇÃO IMPORTANTE ---
// Garante que a pasta de backup exista e tenha permissão de escrita
if (!is_dir($backup_dir)) {
    if (!mkdir($backup_dir, 0755, true)) {
        $feedback = '<div class="alert alert-danger">Falha ao criar a pasta de backups. Verifique as permissões.</div>';
    }
} elseif (!is_writable($backup_dir)) {
     $feedback = '<div class="alert alert-danger">A pasta de backups não tem permissão de escrita. Ajuste as permissões (CHMOD 755 ou 777 se necessário) para a pasta `backups/`.</div>';
}


// --- LÓGICA DE AÇÕES ---

// AÇÃO: Download de Backup
if (isset($_GET['download'])) {
    $file = basename($_GET['download']); // Segurança: basename() evita LFI
    $filepath = $backup_dir . $file;

    if (file_exists($filepath) && is_readable($filepath)) {
        header('Content-Description: File Transfer');
        header('Content-Type: application/octet-stream');
        header('Content-Disposition: attachment; filename="' . $file . '"');
        header('Expires: 0');
        header('Cache-Control: must-revalidate');
        header('Pragma: public');
        header('Content-Length: ' . filesize($filepath));
        flush(); // Limpa o buffer de saída do sistema
        readfile($filepath);
        exit(); // Termina o script
    } else {
        $feedback = '<div class="alert alert-danger">Erro: Arquivo de download não encontrado ou sem permissão de leitura.</div>';
    }
}

// AÇÃO: Deletar Backup
if (isset($_GET['delete'])) {
    $file = basename($_GET['delete']);
    $filepath = $backup_dir . $file;

    if (file_exists($filepath)) {
        if (unlink($filepath)) {
            $feedback = '<div class="alert alert-success">Backup <strong>' . htmlspecialchars($file) . '</strong> deletado com sucesso.</div>';
        } else {
            $feedback = '<div class="alert alert-danger">Erro ao deletar o arquivo <strong>' . htmlspecialchars($file) . '</strong>. Verifique as permissões.</div>';
        }
    } else {
        $feedback = '<div class="alert alert-danger">Erro: Arquivo para deletar não encontrado.</div>';
    }
}

// AÇÕES via POST (Backup e Restore)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {

    // AÇÃO: Criar Novo Backup
    if ($_POST['action'] === 'do_backup') {
        
        if (!is_writable($backup_dir)) {
             $feedback = '<div class="alert alert-danger"><strong>Falha no Backup:</strong> A pasta de backups não tem permissão de escrita. Ajuste as permissões (CHMOD) da pasta `backups/`.</div>';
        } 
        // Verifica se as constantes de DB estão definidas
        elseif (!defined('DB_HOST') || !defined('DB_USER') || !defined('DB_PASS') || !defined('DB_NAME')) {
            $feedback = '<div class="alert alert-danger"><strong>Falha no Backup:</strong> As constantes de conexão com o banco de dados (DB_HOST, DB_USER, DB_PASS, DB_NAME) não estão definidas. Verifique o arquivo `api/controles/db.php`.</div>';
        } 
        else {
            $filename = $backup_dir . 'backup_' . DB_NAME . '_' . date('Y-m-d_H-i-s') . '.sql';
            $exec_success = false; // Flag para controlar o sucesso do exec
            $exec_available = function_exists('exec');

            // [NOVO] TENTATIVA 1: Usar 'exec' (mysqldump) - O método preferido
            if ($exec_available) {
                // Flags importantes:
                // --no-tablespaces: Evita erros de permissão comuns em dumps
                // 2>&1: Redireciona a saída de erro (stderr) para a saída padrão (stdout), para que possamos capturá-la em $output
                $command = sprintf(
                    'mysqldump --host=%s --user=%s --password=%s --no-tablespaces %s > %s 2>&1',
                    escapeshellarg(DB_HOST),
                    escapeshellarg(DB_USER),
                    escapeshellarg(DB_PASS),
                    escapeshellarg(DB_NAME),
                    escapeshellarg($filename)
                );

                @exec($command, $output, $return_var);

                if ($return_var === 0) {
                    // Sucesso com mysqldump
                    if (file_exists($filename) && filesize($filename) > 0) {
                        $feedback = '<div class="alert alert-success">Backup (mysqldump) <strong>' . htmlspecialchars(basename($filename)) . '</strong> criado com sucesso.</div>';
                        $exec_success = true;
                    } elseif (file_exists($filename)) {
                        $feedback = '<div class="alert alert-warning">Backup (mysqldump) <b>' . htmlspecialchars(basename($filename)) . '</b> foi criado, mas está vazio (0 bytes). O banco de dados pode estar vazio.</div>';
                        $exec_success = true; // Tecnicamente foi criado
                    } else {
                        $feedback = '<div class="alert alert-danger">Comando mysqldump executado, mas o arquivo final não foi encontrado. Verifique as permissões e o caminho.</div>';
                    }
                } else {
                    // Falha com exec, anota a falha mas TENTA o método PHP
                    $exec_error_message = implode("\n", $output);
                    $feedback = '<div class="alert alert-warning"><strong>Falha no `mysqldump` (exec):</strong> ' . htmlspecialchars($exec_error_message) . '<br>Tentando backup alternativo em Puro PHP...</div>';
                    if (file_exists($filename)) @unlink($filename); // Apaga arquivo de 0 bytes
                }
            
            } else {
                 // 'exec' não está disponível
                 $feedback = '<div class="alert alert-info">Função `exec` está desabilitada. Tentando backup em "Puro PHP". Isso pode ser lento para bancos de dados grandes.</div>';
            }


            // [NOVO] TENTATIVA 2: Usar "Puro PHP" (se 'exec' falhou ou está desabilitada)
            if (!$exec_success) {
                $pdo = conectar_bd();
                if (!$pdo) {
                    // A função conectar_bd() (do seu db.php) retornou null
                    $feedback .= '<div class="alert alert-danger"><strong>Falha no Backup PHP:</strong> Não foi possível conectar ao banco de dados usando `conectar_bd()`. Verifique se as credenciais estão corretas em `api/controles/db.php`.</div>';
                } else {
                    $result = backup_database_php($pdo, $filename);
                    if ($result === true) {
                        // Se o feedback anterior foi o 'tentando...', sobrescreve com sucesso
                        $feedback = '<div class="alert alert-success">Backup (Puro PHP) <strong>' . htmlspecialchars(basename($filename)) . '</strong> criado com sucesso.</div>';
                    } else {
                        // Adiciona a nova mensagem de erro
                        $feedback .= '<div class="alert alert-danger"><strong>Falha no Backup PHP:</strong> ' . htmlspecialchars($result) . '</div>';
                    }
                }
            }
        }
    }

    // AÇÃO: Restaurar Backup
    if ($_POST['action'] === 'do_restore') {
        
        // [NOVO] VERIFICA SE EXEC ESTÁ HABILITADO ANTES DE TUDO
        if (!function_exists('exec')) {
            $feedback = '<div class="alert alert-danger"><strong>Falha na Restauração:</strong> A função `exec()` do PHP está desabilitada nesta hospedagem. A restauração automática não é possível.<br><br>Por favor, restaure este backup manualmente usando a ferramenta <strong>phpMyAdmin</strong> do seu painel de hospedagem.</div>';
        }
        elseif (!isset($_POST['backup_file']) || empty($_POST['backup_file'])) {
            $feedback = '<div class="alert alert-danger">Nenhum arquivo de backup selecionado para restauração.</div>';
        }
        // Verifica se as constantes de DB estão definidas
        elseif (!defined('DB_HOST') || !defined('DB_USER') || !defined('DB_PASS') || !defined('DB_NAME')) {
            $feedback = '<div class="alert alert-danger"><strong>Falha na Restauração:</strong> As constantes de conexão (DB_HOST, DB_USER, DB_PASS, DB_NAME) não estão definidas.</div>';
        }
        else {
            // *** ESTA É A PARTE QUE FALTAVA ***
            $file = basename($_POST['backup_file']);
            $filepath = $backup_dir . $file;

            if (file_exists($filepath) && is_readable($filepath)) {
                
                $command = sprintf(
                    'mysql --host=%s --user=%s --password=%s %s < %s 2>&1',
                    escapeshellarg(DB_HOST),
                    escapeshellarg(DB_USER),
                    escapeshellarg(DB_PASS),
                    escapeshellarg(DB_NAME),
                    escapeshellarg($filepath)
                );

                @exec($command, $output, $return_var);

                if ($return_var === 0) {
                    $feedback = '<div class="alert alert-success">Banco de dados restaurado com sucesso a partir de <strong>' . htmlspecialchars($file) . '</strong>.</div>';
                } else {
                    $error_message = implode("\n", $output);
                    $feedback = '<div class="alert alert-danger"><strong>Falha ao restaurar o backup.</strong><br>Mensagem: ' . htmlspecialchars($error_message) . '<br><small>Verifique se o `mysql` client está instalado e se as credenciais estão corretas.</small></div>';
                }
            } else {
                $feedback = '<div class="alert alert-danger">Erro: Arquivo de restauração <strong>' . htmlspecialchars($file) . '</strong> não encontrado ou sem permissão de leitura.</div>';
            }
        } // Fim do 'else' (da linha 248)
    } // Fim do 'if ($_POST['action'] === 'do_restore')'
} // Fim do 'if ($_SERVER['REQUEST_METHOD'] === 'POST')'


// Define o título da página para o header
$page_title = "Backup e Restauração";
require_once("menu.php");
?>

<div class="card">
    <div class="card-header bg-primary text-white text-center">
        <h2 class="mb-0"><i class="fas fa-database"></i> Gerenciador de Banco de Dados</h2>
    </div>
    <div class="card-body p-4">
        <?php if ($feedback) echo $feedback; ?>
        
        <div class="text-center mb-4">
            <form method="POST" action="">
                <input type="hidden" name="action" value="do_backup">
                <button type="submit" class="btn btn-lg btn-success"><i class="fas fa-plus-circle"></i> Criar Novo Backup</button>
            </form>
        </div>
        
        <h4 class="mb-3">Backups Existentes</h4>
        <div class="list-group">
            <?php
            // [CORRIGIDO] Lógica de listagem de arquivos mais robusta
            if (!is_dir($backup_dir) || !is_readable($backup_dir)) {
                echo '<div class="alert alert-danger">A pasta de backups (`' . htmlspecialchars($backup_dir) . '`) não existe ou não tem permissão de leitura. Verifique as permissões (CHMOD) da pasta.</div>';
            } else {
                $all_files = scandir($backup_dir, SCANDIR_SORT_DESCENDING); // Lê todos os arquivos e ordena do mais novo para o mais antigo
                $backup_files = array_filter($all_files, function($file) {
                    return pathinfo($file, PATHINFO_EXTENSION) == 'sql';
                });

                if (empty($backup_files)) {
                    echo '<div class="list-group-item">Nenhum backup encontrado.</div>';
                } else {
                    foreach ($backup_files as $file) {
                        $filepath = $backup_dir . $file;
                        echo '<div class="list-group-item d-flex justify-content-between align-items-center flex-wrap gap-2">';
                        echo '  <div class="text-break">';
                        echo '      <i class="fas fa-file-alt text-secondary me-2"></i>';
                        echo '      <strong>' . htmlspecialchars($file) . '</strong>';
                        echo '      <small class="text-muted ms-2">(' . round(filesize($filepath) / 1024 / 1024, 2) . ' MB)</small>';
                        echo '  </div>';
                        echo '  <div class="btn-group" role="group">';
                        echo '      <a href="?download=' . urlencode($file) . '" class="btn btn-sm btn-outline-primary"><i class="fas fa-download"></i> Baixar</a>';
                        echo '      <button class="btn btn-sm btn-outline-warning restore-btn" data-file="' . htmlspecialchars($file) . '"><i class="fas fa-history"></i> Restaurar</button>';
                        echo '      <a href="?delete=' . urlencode($file) . '" class="btn btn-sm btn-outline-danger" onclick="return confirm(\'TEM CERTEZA ABSOLUTA? Esta ação não pode ser desfeita.\');"><i class="fas fa-trash"></i> Apagar</a>';
                        echo '  </div>';
                        echo '</div>';
                    }
                }
            }
            ?>
        </div>
        
        <!-- Formulário oculto para restauração, ativado via JS -->
        <form id="restoreForm" method="POST" action="" style="display: none;">
            <input type="hidden" name="action" value="do_restore">
            <input type="hidden" id="backup_file_input" name="backup_file">
        </form>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    const restoreButtons = document.querySelectorAll('.restore-btn');
    const restoreForm = document.getElementById('restoreForm');
    const backupFileInput = document.getElementById('backup_file_input');
    
    restoreButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            const filename = this.getAttribute('data-file');
            
            // Confirmação dupla para uma ação tão destrutiva
            const confirmation1 = confirm('ALERTA SÉRIO!\n\nVocê tem certeza que quer RESTAURAR o backup: ' + filename + '?\n\N TODOS OS DADOS ATUAIS SERÃO PERDIDOS E SUBSTITUÍDOS PELO BACKUP.');
            
            if (confirmation1) {
                const confirmation2 = confirm('CONFIRMAÇÃO FINAL:\n\nTODOS OS DADOS ATUAIS NO BANCO DE DADOS SERÃO APAGADOS. Continuar com a restauração?');
                
                if (confirmation2) {
                    backupFileInput.value = filename;
                    restoreForm.submit();
                }
            }
        });
    });
});
</script>

<?php
require_once("footer.php");
?>