(() => {
    // =================================================================================
    // CONFIGURAÇÕES GLOBAIS E ESTADO DA APLICAÇÃO
    // =================================================================================
    const URLS_PER_REQUEST = 150; // Quantidade de URLs a serem enviadas por requisição

    let fileContentGlobal = null; // Armazena o conteúdo do arquivo M3U lido
    let isPaused = false;
    let totalUrlsInFile = 0;
    let totalUrlsProcessed = 0;
    let startTime = null;

    // Contadores para o painel de estatísticas
    let counters = {
        totalRequests: 0,
        addedUrls: 0,
        live: 0,
        movies: 0,
        series: 0,
        episodes: 0,
        seasons: 0,
        duplicates: 0,
        errors: 0,
        epg: 0
    };

    // =================================================================================
    // SELETORES DE ELEMENTOS DO DOM (HTML)
    // =================================================================================
    const pauseBtn = document.getElementById("pauseBtn");
    const resumeBtn = document.getElementById("resumeBtn");
    const openFirstModalBtn = document.getElementById('openFirstModal');
    const btnSelectFile = document.getElementById("btnSelectFile");
    const m3uFileInput = document.getElementById("m3uFile");
    const processUrlBtn = document.getElementById("processFileBtn");
    const m3uUrlInput = document.getElementById("m3uUrl");
    const dropArea = document.getElementById("dropArea");
    const dropArea2 = document.getElementById("dropArea2");
    const startImportBtn = document.getElementById('startImportBtn');
    const logContainer = document.getElementById('log-container');
    const categoryListContainer = document.getElementById('categoryList');
    const selectAllCategoriesCheckbox = document.getElementById('selectAllCategories');
    
    // Instâncias dos Modais Bootstrap
    const fileModal = new bootstrap.Modal(document.getElementById("modal_arquivo"));
    const urlModal = new bootstrap.Modal(document.getElementById("modal_url"));
    const categoryModal = new bootstrap.Modal(document.getElementById('modal_categorias_tmdb'));
    const completionModal = new bootstrap.Modal(document.getElementById("completionModal"));

    // =================================================================================
    // FUNÇÃO DE LOG EM TEMPO REAL
    // =================================================================================
    function addToLog(message, type = 'info') {
        const logMessage = document.createElement('p');
        logMessage.className = `log-message ${type}`;
        logMessage.innerHTML = `[${new Date().toLocaleTimeString()}] ${message}`;
        logContainer.appendChild(logMessage);
        logContainer.scrollTop = logContainer.scrollHeight;
    }

    // =================================================================================
    // FUNÇÕES DE PRÉ-PROCESSAMENTO E SELEÇÃO DE CATEGORIAS
    // =================================================================================
    function extractCategories(content) {
        const categories = new Set();
        const regex = /group-title="([^"]+)"/g;
        let match;
        while ((match = regex.exec(content)) !== null) {
            categories.add(match[1].trim());
        }
        return Array.from(categories).sort();
    }

    function showCategorySelectionModal(categories) {
        categoryListContainer.innerHTML = '';
        if (categories.length === 0) {
            categoryListContainer.innerHTML = '<p class="text-center">Nenhuma categoria (group-title) foi encontrada.</p>';
        } else {
            categories.forEach((category, index) => {
                const div = document.createElement('div');
                div.className = 'form-check';
                div.innerHTML = `
                    <input class="form-check-input category-checkbox" type="checkbox" value="${category}" id="category-${index}" checked>
                    <label class="form-check-label" for="category-${index}">${category}</label>
                `;
                categoryListContainer.appendChild(div);
            });
        }
        fileModal.hide();
        urlModal.hide();
        categoryModal.show();
    }
    
    function handleFileReady(content) {
        if (!content.startsWith('#EXTM3U')) {
            addToLog('Erro: O conteúdo fornecido não é um arquivo M3U válido.', 'error');
            Swal.fire("Arquivo Inválido", "O arquivo ou URL não contém um cabeçalho #EXTM3U válido.", "error");
            return;
        }
        
        fileContentGlobal = content;
        const categories = extractCategories(content);
        addToLog(`Arquivo M3U carregado. Encontradas ${categories.length} categorias.`, 'info');
        showCategorySelectionModal(categories);
    }

    // =================================================================================
    // FUNÇÕES PRINCIPAIS DE PROCESSAMENTO E UPLOAD
    // =================================================================================
    function parseM3UContent(content, selectedCategories) {
        const lines = content.split('\n');
        let items = [];
        totalUrlsInFile = 0;

        for (let i = 0; i < lines.length; i++) {
            const line = lines[i].trim();
            if (line.startsWith("#EXTINF")) {
                const groupTitle = getAttribute(line, "group-title");
                if (!selectedCategories.includes(groupTitle)) continue;

                const tvgLogo = getAttribute(line, "tvg-logo");
                const tvgName = getAttribute(line, "tvg-name");
                const tvgId = getAttribute(line, "tvg-id");
                
                let channelName = "Nome não disponível";
                const commaIndex = line.lastIndexOf(",");
                if (commaIndex !== -1) channelName = line.substring(commaIndex + 1).trim();

                const url = lines[i + 1]?.trim();
                if (url && /^(https?|rtsp|ftp):\/\/[^\s]+/g.test(url)) {
                    items.push({ url, groupTitle, tvgLogo, tvgName, tvgId, channelName });
                    totalUrlsInFile++;
                }
            }
        }
        return items;
    }

    async function sendChunkToServer(chunk, useTmdb) {
        chunk.forEach(item => {
            addToLog(`Processando: <strong>${item.channelName}</strong>`, 'info');
        });

        const dataToSend = { use_tmdb: useTmdb, items: chunk };
    
        try {
            // ✨ MUDANÇA PRINCIPAL AQUI ✨
            const response = await fetch("./api/controles/importar-arquivo2025-m3u.php", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify(dataToSend)
            });
    
            if (!response.ok) {
                const errorText = await response.text();
                throw new Error(`Erro ${response.status}: ${errorText}`);
            }
    
            counters.totalRequests++;
            const result = await response.json();
    
            if (result.results && typeof result.results === "object") {
                const { success, exists, error, urls, movie, series, live, episodios, temporadas } = result.results;
                
                if (Array.isArray(success) && success.length > 0) {
                     success.forEach(item => addToLog(`Sucesso: '<strong>${item.name}</strong>' importado como ${item.type}.`, 'success'));
                }
                if (Array.isArray(error) && error.length > 0) {
                     error.forEach(item => addToLog(`Erro: '<strong>${item.name}</strong>' - ${item.reason}`, 'error'));
                }

                if (urls) counters.addedUrls += urls;
                if (movie) counters.movies += movie;
                if (series) counters.series += series;
                if (live) counters.live += live;
                if (episodios) counters.episodes += episodios;
                if (temporadas) counters.seasons += temporadas;
                if (exists) counters.duplicates += exists;
                if (error) counters.errors += error.length;
                
                totalUrlsProcessed += chunk.length;
            }
    
        } catch (error) {
            console.error("Erro ao enviar bloco:", error);
            addToLog(`Falha crítica ao enviar bloco: ${error.message}`, 'error');
            counters.errors += chunk.length;
            totalUrlsProcessed += chunk.length;
        }
    }

    async function startImportProcess(selectedCategories, useTmdb) {
        categoryModal.hide();
        addToLog(`Iniciando importação...`, 'info');
        
        document.getElementById("openFirstModal").style.display = "none";
        document.getElementById("controles").style.display = "block";
        startTime = new Date();
        
        const itemsToProcess = parseM3UContent(fileContentGlobal, selectedCategories);
        const chunks = chunkArray(itemsToProcess, URLS_PER_REQUEST);

        addToLog(`Processando <strong>${itemsToProcess.length}</strong> itens da lista. Dividido em <strong>${chunks.length}</strong> blocos.`, 'info');
        
        totalUrlsProcessed = 0;
        for (const chunk of chunks) {
            while (isPaused) {
                await new Promise(resolve => setTimeout(resolve, 1000));
            }
            await sendChunkToServer(chunk, useTmdb);
            updateAllUI();
        }
        
        addToLog("Processo de importação finalizado!", "info");
        await processEpg();
        completionModal.show();
        document.getElementById("controles").style.display = "none";
    }

    // =================================================================================
    // FUNÇÕES DE EPG, UTILITÁRIAS e EVENT LISTENERS
    // =================================================================================
    async function processEpg() {
        try {
            Swal.fire({ title: "Verificando EPG", html: "Aguarde...", icon: "info", showConfirmButton: false, allowOutsideClick: false, timerProgressBar: true, didOpen: () => { Swal.showLoading() } });
            const response = await fetch("./xmltv.php?epg");
            if (!response.ok) throw new Error(`Erro no servidor: ${response.statusText}`);
            const xmlText = await response.text();
            if (!xmlText.startsWith("<?xml")) throw new Error("A resposta não é um XML válido.");
            const parser = new DOMParser();
            const xmlDoc = parser.parseFromString(xmlText, "text/xml");
            if (xmlDoc.getElementsByTagName("parsererror").length > 0) throw new Error("XML malformado.");
            const channels = xmlDoc.getElementsByTagName("channel");
            if (channels.length === 0) { Swal.close(); return; }
            let epgChannels = [];
            for (let channel of channels) {
                let id = channel.getAttribute("id");
                let name = channel.getElementsByTagName("display-name")[0]?.textContent || "Sem Nome";
                if (id) epgChannels.push({ id, name });
            }
            await Swal.update({ title: "Processando EPG...", html: `Encontrados ${epgChannels.length} canais. Atualizando...` });
            const chunks = chunkArray(epgChannels, 500);
            for (const chunk of chunks) { await sendEpgChunk(chunk); }
            await Swal.fire("Sucesso!", `EPG atualizado com ${epgChannels.length} canais.`, "success");
        } catch (error) {
            console.error("Erro no EPG:", error);
            await Swal.fire("Erro no EPG", `<small>${error.message}</small>`, "error");
        }
    }
    async function sendEpgChunk(chunk) {
        const formData = new FormData();
        chunk.forEach((epg, index) => {
            formData.append(`epg[${index}][id]`, epg.id);
            formData.append(`epg[${index}][nome]`, epg.name);
        });
        try {
            // ✨ MUDANÇA PRINCIPAL AQUI TAMBÉM ✨
            const response = await fetch("./api/controles/importar-arquivo2025-m3u.php", { method: "POST", body: formData });
            if (!response.ok) throw new Error(`Erro HTTP: ${response.status}`);
            const result = await response.json();
            if (result.results?.epg) { counters.epg += result.results.epg; updateAllUI(); }
        } catch (error) { console.error("Erro ao enviar bloco de EPG:", error); }
    }
    function updateAllUI() {
        document.getElementById("totalRequests").textContent = counters.totalRequests;
        document.getElementById("add_urls").textContent = counters.addedUrls;
        document.getElementById("canais").textContent = counters.live;
        document.getElementById("filmes").textContent = counters.movies;
        document.getElementById("series_adicionando").textContent = counters.series;
        document.getElementById("episodios_adicionando").textContent = counters.episodes;
        document.getElementById("temporadas_adicionando").textContent = counters.seasons;
        document.getElementById("exitente").textContent = counters.duplicates;
        document.getElementById("Erro").textContent = counters.errors;
        document.getElementById("epg_adicionando").textContent = counters.epg;
        if (totalUrlsInFile > 0) {
            const progress = (totalUrlsProcessed / totalUrlsInFile) * 100;
            const progressBar = document.getElementById("partProgressBar");
            progressBar.style.width = `${Math.min(progress, 100)}%`;
            progressBar.textContent = `${Math.min(progress, 100).toFixed(2)}%`;
        }
        if (startTime && totalUrlsProcessed > 0) {
            const elapsedSeconds = (new Date() - startTime) / 1000;
            const timePerUrl = elapsedSeconds / totalUrlsProcessed;
            const estimatedTotalTime = timePerUrl * totalUrlsInFile;
            const remainingTime = estimatedTotalTime - elapsedSeconds;
            document.getElementById("tempo_Total_Estimado").textContent = formatTime(estimatedTotalTime);
            document.getElementById("tempo_Decorrido").textContent = formatTime(elapsedSeconds);
            document.getElementById("tempo_Restante").textContent = formatTime(remainingTime > 0 ? remainingTime : 0);
        }
    }
    function formatTime(seconds) {
        const h = Math.floor(seconds / 3600);
        const m = Math.floor((seconds % 3600) / 60);
        const s = Math.floor(seconds % 60);
        return `${String(h).padStart(2, '0')}:${String(m).padStart(2, '0')}:${String(s).padStart(2, '0')}`;
    }
    function chunkArray(arr, size) {
        const chunks = [];
        for (let i = 0; i < arr.length; i += size) chunks.push(arr.slice(i, i + size));
        return chunks;
    }
    function getAttribute(line, attributeName) {
        const regex = new RegExp(`${attributeName}="([^"]+)"`);
        const match = line.match(regex);
        return match ? match[1] : "";
    }
    btnSelectFile.addEventListener("click", () => m3uFileInput.click());
    m3uFileInput.addEventListener("change", (event) => {
        const file = event.target.files[0];
        if (file) {
            addToLog(`Arquivo local selecionado: ${file.name}`, 'info');
            const reader = new FileReader();
            reader.onload = (e) => handleFileReady(e.target.result);
            reader.readAsText(file);
        }
    });
    processUrlBtn.addEventListener("click", () => {
        const url = m3uUrlInput.value;
        if (!url) { Swal.fire("URL Vazia", "Por favor, insira a URL.", "warning"); return; }
        addToLog(`Buscando arquivo da URL: ${url}`, 'info');
        processUrlBtn.disabled = true;
        fetch(`./api/proxy.php?url=${encodeURIComponent(url)}`)
            .then(response => { if (!response.ok) throw new Error(`Erro no proxy: ${response.statusText}`); return response.text(); })
            .then(content => { handleFileReady(content); })
            .catch(error => { console.error("Erro ao buscar URL:", error); addToLog(`Falha ao buscar URL: ${error.message}`, 'error'); Swal.fire("Erro", "Não foi possível baixar o arquivo da URL.", "error"); })
            .finally(() => { processUrlBtn.disabled = false; });
    });
    startImportBtn.addEventListener('click', () => {
        const selectedCategories = Array.from(document.querySelectorAll('.category-checkbox:checked')).map(cb => cb.value);
        if (selectedCategories.length === 0) { Swal.fire("Atenção", "Selecione pelo menos uma categoria.", "warning"); return; }
        const useTmdb = document.getElementById('useTmdb').checked;
        startImportProcess(selectedCategories, useTmdb);
    });
    pauseBtn.addEventListener("click", () => { isPaused = true; pauseBtn.disabled = true; resumeBtn.disabled = false; addToLog("Processo pausado.", "warning"); });
    resumeBtn.addEventListener("click", () => { isPaused = false; pauseBtn.disabled = false; resumeBtn.disabled = true; addToLog("Processo retomado.", "info"); });
    selectAllCategoriesCheckbox.addEventListener('change', function() { document.querySelectorAll('.category-checkbox').forEach(checkbox => checkbox.checked = this.checked); });
    document.getElementById("openFirstModal").addEventListener("click", () => fileModal.show());
    document.getElementById("openSecondModal").addEventListener("click", () => { fileModal.hide(); urlModal.show(); });
    document.getElementById("backToFirstModal").addEventListener("click", () => { urlModal.hide(); fileModal.show(); });
    ['dragenter', 'dragover', 'dragleave', 'drop'].forEach(eventName => { document.body.addEventListener(eventName, e => { e.preventDefault(); e.stopPropagation(); }); });
    document.body.addEventListener('dragenter', () => { dropArea.style.display = "block"; dropArea2.style.display = "none"; });
    document.body.addEventListener('dragleave', (e) => { if (!e.relatedTarget) { dropArea.style.display = "none"; dropArea2.style.display = "block"; } });
    document.body.addEventListener('drop', (e) => {
        dropArea.style.display = "none";
        dropArea2.style.display = "block";
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            const file = files[0];
            if (file.name.endsWith('.m3u')) {
                const reader = new FileReader(); reader.onload = (e) => handleFileReady(e.target.result); reader.readAsText(file);
            } else { Swal.fire('Arquivo Inválido', 'Por favor, solte um arquivo .m3u', 'warning'); }
        }
    });
    function init() {
        const autor1 = document.getElementById("autor1");
        const autor2 = document.getElementById("autor2");
        if (autor1) autor1.innerHTML = '<a href="https://t.me/Dev_Base" target="_blank">Feito por: 💡 Dev_Base</a>';
        if (autor2) autor2.innerHTML = '<a href="https://t.me/Dev_Base" target="_blank">Feito por: 💡 Dev_Base</a>';
        fileModal.show();
        logContainer.innerHTML = '';
        addToLog('Sistema pronto. Selecione um arquivo ou insira uma URL para começar.', 'info');
        document.getElementById("controles").style.display = "none";
    }
    init();
})();