document.addEventListener('DOMContentLoaded', () => {
    // --- LÓGICA DO TEMA CLARO/ESCURO ---
    const themeToggle = document.getElementById('theme-toggle');
    if (themeToggle) {
        const themeToggleIcon = themeToggle.querySelector('i');
        
        const applyTheme = (theme) => {
            document.documentElement.setAttribute('data-theme', theme);
            localStorage.setItem('theme', theme);
            if (theme === 'dark') {
                themeToggleIcon.classList.remove('bi-moon-stars-fill');
                themeToggleIcon.classList.add('bi-sun-fill');
            } else {
                themeToggleIcon.classList.remove('bi-sun-fill');
                themeToggleIcon.classList.add('bi-moon-stars-fill');
            }
        };

        const currentTheme = localStorage.getItem('theme');
        if (currentTheme) {
            applyTheme(currentTheme);
        } else if (window.matchMedia && window.matchMedia('(prefers-color-scheme: dark)').matches) {
            applyTheme('dark');
        }

        themeToggle.addEventListener('click', () => {
            const newTheme = document.documentElement.getAttribute('data-theme') === 'dark' ? 'light' : 'dark';
            applyTheme(newTheme);
        });
    }

    // --- INICIALIZAÇÃO DE ELEMENTOS DO PAINEL ---
    const tabelaRevendedores = document.getElementById('tabela-revendedores');
    const campoBusca = document.getElementById('campo-busca');
    const revendedorModalEl = document.getElementById('revendedorModal');
    const revendedorModal = revendedorModalEl ? new bootstrap.Modal(revendedorModalEl) : null;
    const formRevendedor = document.getElementById('form-revendedor');
    const btnSalvarRevendedor = document.getElementById('btnSalvarRevendedor');
    const creditosModalEl = document.getElementById('creditosModal');
    const creditosModal = creditosModalEl ? new bootstrap.Modal(creditosModalEl) : null;
    const formCreditos = document.getElementById('form-creditos');
    const btnConfirmarCreditos = document.getElementById('btnConfirmarCreditos');
    const creditosConfirmacaoModal = document.getElementById('creditosConfirmacaoModal') ? new bootstrap.Modal(document.getElementById('creditosConfirmacaoModal')) : null;
    const mensagemCreditosTexto = document.getElementById('mensagem-creditos-texto');
    const btnCopiarCreditos = document.getElementById('btn-copiar-creditos');
    const btnWhatsappCreditos = document.getElementById('btn-whatsapp-creditos');
    const revendedorCriadoModal = document.getElementById('revendedorCriadoModal') ? new bootstrap.Modal(document.getElementById('revendedorCriadoModal')) : null;
    const mensagemRevendedorTexto = document.getElementById('mensagem-revendedor-texto');
    const btnCopiarRevendedor = document.getElementById('btn-copiar-revendedor');
    const btnWhatsappRevendedor = document.getElementById('btn-whatsapp-revendedor');
    
    // --- FUNÇÃO AUXILIAR PARA BOTÕES DINÂMICOS (SPINNER) ---
    const toggleButtonSpinner = (button, show) => {
        if (!button) return;
        if (show) {
            button.disabled = true;
            button.dataset.originalText = button.innerHTML;
            button.innerHTML = `<span class="spinner-border spinner-border-sm" role="status" aria-hidden="true"></span>`;
        } else {
            button.disabled = false;
            if (button.dataset.originalText) {
                button.innerHTML = button.dataset.originalText;
            }
        }
    };

    // --- FUNÇÃO PARA CARREGAR A LISTA ---
    const carregarRevendedores = async () => {
        try {
            const response = await fetch('api/revendedores/listar.php');
            if (!response.ok) { throw new Error(`Erro de rede ou servidor: ${response.status}`); }
            const result = await response.json();
            if (result.status !== 'success') { throw new Error(result.message || 'Erro ao buscar dados.'); }

            tabelaRevendedores.innerHTML = '';
            if (result.data.length > 0) {
                result.data.forEach(rev => {
                    // CORRIGIDO: Define uma variável para o nome. Se rev.nome for nulo ou vazio, usa rev.user.
                    const nomeDisplay = rev.nome || rev.user;

                    tabelaRevendedores.innerHTML += `
                        <tr class="revendedor-row">
                            <td class="revendedor-nome">${nomeDisplay}</td>
                            <td class="revendedor-usuario">${rev.user}</td>
                            <td>${parseFloat(rev.creditos).toFixed(2)}</td>
                            <td><span class="badge bg-success">${rev.status}</span></td>
                            <td class="text-center">
                                <button class="btn btn-sm btn-outline-success btn-creditos" data-id="${rev.id}" data-nome="${nomeDisplay}" data-bs-toggle="tooltip" title="Adicionar Créditos"><i class="bi bi-coin"></i></button>
                                <button class="btn btn-sm btn-outline-warning btn-editar" data-id="${rev.id}" data-bs-toggle="tooltip" title="Editar Revendedor"><i class="bi bi-pencil-square"></i></button>
                                <button class="btn btn-sm btn-outline-danger btn-excluir" data-id="${rev.id}" data-bs-toggle="tooltip" title="Excluir Revendedor"><i class="bi bi-trash-fill"></i></button>
                            </td>
                        </tr>`;
                });
            } else {
                tabelaRevendedores.innerHTML = '<tr><td colspan="5" class="text-center">Nenhum revendedor encontrado.</td></tr>';
            }
            const tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
            tooltipTriggerList.map(tooltipTriggerEl => new bootstrap.Tooltip(tooltipTriggerEl));
        } catch(error) {
            console.error("ERRO ao carregar revendedores:", error);
            tabelaRevendedores.innerHTML = `<tr><td colspan="5" class="text-center text-danger">Falha ao carregar a lista. Detalhe: ${error.message}</td></tr>`;
        }
    };

    // --- EVENT LISTENERS ---
    
    if(formRevendedor) {
        formRevendedor.addEventListener('submit', async (e) => {
            e.preventDefault();
            toggleButtonSpinner(btnSalvarRevendedor, true);
            const id = document.getElementById('revendedorId').value;
            const url = id ? 'api/revendedores/atualizar.php' : 'api/revendedores/criar.php';
            const dados = { id: id || undefined, nome: document.getElementById('nome-revendedor').value, usuario: document.getElementById('usuario-revendedor').value, senha: document.getElementById('senha-revendedor').value };
            try {
                const response = await fetch(url, { method: 'POST', headers: {'Content-Type': 'application/json'}, body: JSON.stringify(dados) });
                const result = await response.json();
                if (result.status === 'success') {
                    if(revendedorModal) revendedorModal.hide();
                    await carregarRevendedores();
                    if (!id && result.dados_mensagem) {
                        const { usuario, senha } = result.dados_mensagem;
                        const linkPainel = `${window.location.protocol}//${window.location.hostname}`;
                        const mensagem = `Seja bem-vindo(a)!\n\nDados de acesso:\nLink: ${linkPainel}\nUsuário: ${usuario}\nSenha: ${senha}`;
                        mensagemRevendedorTexto.textContent = mensagem;
                        if(revendedorCriadoModal) revendedorCriadoModal.show();
                    } else { alert(result.message); }
                } else { alert(result.message); }
            } catch(error) { console.error("Erro ao salvar revendedor:", error); } 
            finally { toggleButtonSpinner(btnSalvarRevendedor, false); }
        });
    }

    if(tabelaRevendedores) {
        tabelaRevendedores.addEventListener('click', async (e) => {
            const target = e.target.closest('button');
            if (!target) return;
            const id = target.dataset.id;
            if (target.classList.contains('btn-creditos')) {
                document.getElementById('creditosRevendedorId').value = id;
                document.getElementById('nomeRevendedorCreditos').textContent = target.dataset.nome;
                if(creditosModal) creditosModal.show();
            } else if (target.classList.contains('btn-editar')) {
                const response = await fetch(`api/revendedores/buscar_um.php?id=${id}`);
                const result = await response.json();
                if (result.status === 'success') {
                    document.getElementById('revendedorModalTitulo').textContent = 'Editar Revendedor';
                    document.getElementById('revendedorId').value = result.data.id;
                    document.getElementById('nome-revendedor').value = result.data.nome;
                    document.getElementById('usuario-revendedor').value = result.data.user;
                    document.getElementById('senha-revendedor').value = '';
                    if(revendedorModal) revendedorModal.show();
                } else { alert(result.message); }
            } else if (target.classList.contains('btn-excluir')) {
                if (confirm(`Tem certeza que deseja excluir o revendedor?`)) {
                    const response = await fetch('api/revendedores/excluir.php', { method: 'POST', headers: {'Content-Type': 'application/json'}, body: JSON.stringify({id}) });
                    const result = await response.json();
                    alert(result.message);
                    if (result.status === 'success') { carregarRevendedores(); }
                }
            }
        });
    }

    if(formCreditos) {
        formCreditos.addEventListener('submit', async (e) => {
            e.preventDefault();
            toggleButtonSpinner(btnConfirmarCreditos, true);
            const dados = { id: document.getElementById('creditosRevendedorId').value, creditos: document.getElementById('quantidade-creditos').value };
            try {
                const response = await fetch('api/revendedores/adicionar_credito.php', { method: 'POST', headers: {'Content-Type': 'application/json'}, body: JSON.stringify(dados) });
                const result = await response.json();
                if (result.status === 'success') {
                    if(creditosModal) creditosModal.hide();
                    await carregarRevendedores();
                    const { nome_revendedor, creditos_adicionados } = result.dados_mensagem;
                    const valor = parseFloat(creditos_adicionados);
                    const acao = valor >= 0 ? 'adicionado' : 'removido';
                    const valorAbsoluto = Math.abs(valor);
                    const mensagem = `✅ Operação Concluída!\n\nOlá, ${nome_revendedor}!\nFoi ${acao} ${valorAbsoluto.toFixed(2)} crédito(s) em sua conta.`;
                    mensagemCreditosTexto.textContent = mensagem;
                    if(creditosConfirmacaoModal) creditosConfirmacaoModal.show();
                } else { alert(result.message); }
            } catch(error) { console.error("Erro ao adicionar créditos:", error); } 
            finally { toggleButtonSpinner(btnConfirmarCreditos, false); }
        });
    }

    const setupCopyAndWhatsapp = (modalEl, btnCopy, btnWpp, textSource) => {
        if(!modalEl || !btnCopy || !btnWpp || !textSource) return;
        btnCopy.addEventListener('click', () => { navigator.clipboard.writeText(textSource.textContent).then(() => { const o = btnCopy.innerHTML; btnCopy.innerHTML = '<i class="bi bi-check-lg"></i> Copiado!'; setTimeout(() => { btnCopy.innerHTML = o; }, 2000); }); });
        btnWpp.addEventListener('click', () => { window.open(`https://api.whatsapp.com/send?text=${encodeURIComponent(textSource.textContent)}`, '_blank'); });
    };
    setupCopyAndWhatsapp(document.getElementById('creditosConfirmacaoModal'), btnCopiarCreditos, btnWhatsappCreditos, mensagemCreditosTexto);
    setupCopyAndWhatsapp(document.getElementById('revendedorCriadoModal'), btnCopiarRevendedor, btnWhatsappRevendedor, mensagemRevendedorTexto);
    
    if(revendedorModalEl) revendedorModalEl.addEventListener('hidden.bs.modal', () => { formRevendedor.reset(); document.getElementById('revendedorId').value = ''; document.getElementById('revendedorModalTitulo').textContent = 'Novo Revendedor'; });
    if(creditosModalEl) creditosModalEl.addEventListener('hidden.bs.modal', () => formCreditos.reset());

    // --- CARGA INICIAL ---
    carregarRevendedores();
});