<?php
// ==========================================================
// ARQUIVO FINAL COM PROTEÇÃO DE SAÍDA (Output Buffering)
// Isso impede que espaços em branco acidentais corrompam o JSON.
// ==========================================================

// <<-- PASSO 1: Inicia o buffer de saída.
// Isso captura toda a saída (incluindo espaços em branco)
ob_start();

// Configurações de produção
ini_set('display_errors', 0);
error_reporting(0);
set_time_limit(600); // 10 minutos
ini_set('memory_limit', '512M'); 

require_once(__DIR__ . '/controles/db.php');
session_start();

// ==========================================================
// FUNÇÃO DE RESPOSTA JSON (A Prova de Erros)
// ==========================================================
function sendResponse($status, $message, $http_code = 200) {
    // <<-- PASSO 2: Limpa o buffer de qualquer "lixo" (espaços)
    ob_end_clean(); 
    
    // Envia os cabeçalhos corretos
    http_response_code($http_code);
    header('Content-Type: application/json; charset=utf-8');
    
    // Envia a resposta JSON limpa
    echo json_encode(['status' => $status, 'message' => $message]);
    exit;
}

// Função de Erro (agora usa a sendResponse)
function sendError($message, $http_code = 400) {
    // Envia o erro usando a função segura
    sendResponse('error', $message, $http_code);
}
// ==========================================================


// --- VERIFICAÇÃO DE SEGURANÇA (TOKEN) ---
$conexao = conectar_bd();
if (!$conexao) {
    sendError('Falha ao conectar ao banco de dados.', 500);
}
$conexao->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
$conexao->exec("SET NAMES 'utf8mb4'");

$data = json_decode(file_get_contents('php://input'), true);
if (json_last_error() !== JSON_ERROR_NONE) {
    $data = $_POST; // Fallback
}

$token_enviado = $data['token'] ?? '';
if (empty($token_enviado)) {
    sendError('Acesso negado: Token de autenticação não fornecido.', 403);
}

try {
    // Usa a coluna 'admin'
    $sql_token = "SELECT id, admin FROM admin WHERE token = :token";
    $stmt_token = $conexao->prepare($sql_token);
    $stmt_token->bindParam(':token', $token_enviado, PDO::PARAM_STR);
    $stmt_token->execute();
    $admin_logado = $stmt_token->fetch(PDO::FETCH_ASSOC);

    if (!$admin_logado || empty($admin_logado['id'])) {
        sendError('Acesso negado: Token inválido ou expirado.', 403);
    }
    
    if ($admin_logado['admin'] != 1) {
          sendError('Acesso negado: Apenas o Administrador principal pode executar esta ação.', 403);
    }

} catch (PDOException $e) {
    sendError('Erro no banco de dados durante a validação do token. (' . $e->getMessage() . ')', 500);
}

// ===================================================================
// LÓGICA PRINCIPAL DE LIMPEZA
// ===================================================================
try {
    $conexao->beginTransaction();

    // 1. Limpar Tabela 'streams' (Filmes e Canais)
    $sql_streams = "
        DELETE t1 
        FROM streams AS t1
        INNER JOIN streams AS t2
        WHERE
            t1.id > t2.id AND
            t1.name = t2.name AND
            t1.link = t2.link;
    ";
    $stmt_streams = $conexao->prepare($sql_streams);
    $stmt_streams->execute();
    $count_streams = $stmt_streams->rowCount();

    // 2. Limpar Tabela 'series_episodes' (Episódios)
    $sql_episodes = "
        DELETE t1 
        FROM series_episodes AS t1
        INNER JOIN series_episodes AS t2
        WHERE
            t1.id > t2.id AND
            t1.title = t2.title AND
            t1.link = t2.link;
    ";
    
    $stmt_episodes = $conexao->prepare($sql_episodes);
    $stmt_episodes->execute();
    $count_episodes = $stmt_episodes->rowCount();

    $conexao->commit();

    $message = "Limpeza concluída com sucesso!\n\n";
    $message .= "Filmes/Canais duplicados removidos: $count_streams\n";
    $message .= "Episódios duplicados removidos: $count_episodes\n";

    // <<-- PASSO 3: Envia a resposta de sucesso usando a função segura
    sendResponse('success', $message);

} catch (Throwable $t) { 
    if ($conexao->inTransaction()) {
        $conexao->rollBack();
    }
    // <<-- PASSO 4: Envia o erro usando a função segura
    sendError('Erro fatal no servidor durante a limpeza: ' . $t->getMessage(), 500);
}
?>