<?php
// Arquivo: /api/controles/funcoes_testes.php
// VERSÃO 19 - 100% LIMPA E CORRIGIDA

/**
* Encurta uma URL longa usando a API is.gd.
 */
if (!function_exists('encurtar_url')) {
    function encurtar_url($long_url) {
        if (empty($long_url) || !filter_var($long_url, FILTER_VALIDATE_URL)) return $long_url;
        $api_url = 'https://is.gd/api.php?longurl=' . urlencode($long_url);
        if (function_exists('curl_init')) {
            $ch = curl_init();
            curl_setopt_array($ch, [CURLOPT_URL => $api_url, CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => 5]);
            $short_url = curl_exec($ch);
            curl_close($ch);
        } else {
            $short_url = @file_get_contents($api_url);
        }
        return ($short_url && filter_var(trim($short_url), FILTER_VALIDATE_URL)) ? trim($short_url) : $long_url;
    }
}

/**
* Obtem uma URL aleatoria de um servidor de Load Balance ativo.
 */
if (!function_exists('obter_url_load_balance_ou_padrao')) {
    function obter_url_load_balance_ou_padrao($conexao) {
        // URL padrao caso o load balance falhe
        $portal_url_padrao = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://" . $_SERVER['HTTP_HOST'];
        
        try {
            // 1. Seleciona o IP e a PORTA.
            $stmt_servers = $conexao->prepare("SELECT server_ip, server_port FROM servers WHERE status = 1");
            
            if (!$stmt_servers || !$stmt_servers->execute()) {
                 error_log("Erro ao consultar a tabela 'servers' para Load Balance.");
                 return $portal_url_padrao;
            }

            // 2. Busca como array associativo
            $servidores_ativos = $stmt_servers->fetchAll(PDO::FETCH_ASSOC);

            if (empty($servidores_ativos)) {
                 error_log("Nenhum servidor de load balance ativo (status=1) encontrado. Usando URL padrao.");
                 return $portal_url_padrao;
            }

            // 3. Seleciona um servidor aleatorio
            $servidor_selecionado = $servidores_ativos[array_rand($servidores_ativos)];
            
            $portal_url = rtrim($servidor_selecionado['server_ip'], '/');
            $port = $servidor_selecionado['server_port']; 

            // 4. Garante que a URL tem o protocolo
            if (strpos($portal_url, 'http://') !== 0 && strpos($portal_url, 'https://') !== 0) {
                 // Se a porta for 443, assume https, senao http
                 if ($port == 443) {
                     $portal_url = 'https://' . $portal_url;
                 } else {
                     $portal_url = 'http://' . $portal_url;
                 }
            }

            // 5. Adiciona a porta
            if (!empty($port) && is_numeric($port) && $port != 80 && $port != 443) {
                 $portal_url = rtrim($portal_url, '/'); 
                 $portal_url .= ':' . $port;
            }
            
            return $portal_url; 

        } catch (Exception $e) {
             error_log("Erro Critico ao buscar DNS (obter_url_load_balance_ou_padrao): " . $e->getMessage());
             return $portal_url_padrao;
        }
    }
}


/**
* Lista os clientes com is_trial = 1 para o DataTable
 */
function listar_testes($dados_requisicao) {
    try {
        $conexao = conectar_bd();
        $admin_id = $_SESSION['admin_id'];
        $colunas = [0 => 'id', 1 => 'name', 2 => 'usuario', 4 => 'Vencimento'];

        $query_base = " FROM clientes WHERE admin_id = :admin_id AND is_trial = 1";
        $params = [':admin_id' => $admin_id];

        if (!empty($dados_requisicao['search']['value'])) {
            $search_value = "%" . $dados_requisicao['search']['value'] . "%";
            $query_base .= " AND (name LIKE :search_name OR usuario LIKE :search_usuario)";
            $params[':search_name'] = $search_value;
            $params[':search_usuario'] = $search_value;
        }

        $stmt_total = $conexao->prepare("SELECT COUNT(id) FROM clientes WHERE admin_id = :admin_id AND is_trial = 1");
        $stmt_total->execute([':admin_id' => $admin_id]);
        $total_records = $stmt_total->fetchColumn();

        $stmt_filtrado = $conexao->prepare("SELECT COUNT(id) " . $query_base);
        $stmt_filtrado->execute($params);
        $total_filtered = $stmt_filtrado->fetchColumn();

        $coluna_ordenacao = $colunas[$dados_requisicao['order'][0]['column']] ?? 'id';
        $direcao_ordenacao = $dados_requisicao['order'][0]['dir'] ?? 'desc';
        $inicio = (int)($dados_requisicao['start'] ?? 0);
        $quantidade = (int)($dados_requisicao['length'] ?? 10);
        
        $query_final = "SELECT id, name, usuario, Vencimento " . $query_base . " ORDER BY " . $coluna_ordenacao . " " . $direcao_ordenacao . " LIMIT " . $inicio . ", " . $quantidade;
        
        $stmt_principal = $conexao->prepare($query_final);
        $stmt_principal->execute($params);

        $dados = [];
        while ($row = $stmt_principal->fetch(PDO::FETCH_ASSOC)) {
            $id = $row['id'];
            $status = (strtotime($row['Vencimento']) < time()) ? '<span class="badge bg-warning text-dark">Expirado</span>' : '<span class="badge bg-info">Ativo</span>';
            $acoes = "<div class='btn-group' role='group'><button type='button' class='btn btn-sm btn-info' onclick='modal_master(\"api/testes.php\", \"info_teste\", \"{$id}\")' title='Info'><i class='fas fa-eye'></i></button><button type='button' class='btn btn-sm btn-success' onclick='modal_master(\"api/testes.php\", \"converter_teste\", \"{$id}\")' title='Converter'><i class='fas fa-check'></i></button><button type='button' class='btn btn-sm btn-danger' onclick='modal_master(\"api/testes.php\", \"delete_teste\", \"{$id}\")' title='Excluir'><i class='fas fa-trash-alt'></i></button></div>";
            $dados[] = ["id" => $id, "name" => htmlspecialchars($row['name']), "usuario" => htmlspecialchars($row['usuario']), "status" => $status, "vencimento" => date('d/m/Y H:i', strtotime($row['Vencimento'])), "acao" => $acoes];
        }
        
        return ["draw" => intval($dados_requisicao['draw']), "recordsTotal" => intval($total_records), "recordsFiltered" => intval($total_filtered), "data" => $dados];

    } catch (PDOException $e) {
        error_log("Erro no DataTables (listar_testes): " . $e->getMessage());
        return ["error" => "Erro de Banco de Dados: " . $e->getMessage(), "data" => []];
    }
}

function info_teste($id) {
    try {
        $conexao = conectar_bd();
        $admin_id = $_SESSION['admin_id'] ?? 0;
        $stmt = $conexao->prepare("SELECT * FROM clientes WHERE id = :id AND admin_id = :admin_id");
        $stmt->execute([':id' => $id, ':admin_id' => $admin_id]);

        if ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            extract($row);
            $template_file = $_SERVER['DOCUMENT_ROOT'] . '/template_mensagem.txt';
            $template = file_exists($template_file) ? file_get_contents($template_file) : "ERRO: template_mensagem.txt nao encontrado!";
            
            $portal_url = obter_url_load_balance_ou_padrao($conexao);
            
            $exp_date_formatted = date('d/m/Y H:i:s', strtotime($Vencimento));

            $pretty_m3u_ts = $portal_url . '/m3u-ts/' . $usuario . '/' . $senha;
            $pretty_m3u_hls = $portal_url . '/m3u-m3u8/' . $usuario . '/' . $senha;
            $pretty_ssiptv = $portal_url . '/ss-ts/' . $usuario . '/' . $senha;

            $replacements = [
                '#username#' => $usuario, '#password#' => $senha, '#url#' => $portal_url, '#exp_date#' => $exp_date_formatted,
                '#m3u_link#' => $portal_url . '/get.php?username=' . $usuario . '&password=' . $senha . '&type=m3u_plus&output=ts', 
                '#m3u_link_hls#' => $portal_url . '/get.php?username=' . $usuario . '&password=' . $senha . '&type=m3u_plus&output=m3u8',
                '#ssiptv_link#' => $portal_url . '/get.php?username=' . $usuario . '&password=' . $senha . '&type=m3u_plus&output=ts',
                '#m3u_encurtado#' => $pretty_m3u_ts, 
                '#m3u_hls_encurtado#' => $pretty_m3u_hls, 
                '#ssiptv_encurtado#' => $pretty_ssiptv,
            ];
            $mensagem_final = str_replace(array_keys($replacements), array_values($replacements), $template);
            
            // --- INÍCIO DA CORREÇÃO ---
            // Variáveis preparadas para o Heredoc
            $element_id = "infoClienteTexto-" . $id;
            $mensagem_copy = htmlspecialchars($mensagem_final, ENT_QUOTES);
            $mensagem_html = htmlspecialchars($mensagem_final);
            $mensagem_url = urlencode($mensagem_final);

            $modal_body = <<<HTML
            <div id="{$element_id}"
                 data-copy-text="{$mensagem_copy}"
                 data-raw-text="{$mensagem_copy}"
                 class="form-control" readonly style="font-family: monospace; font-size: 0.9rem; resize: none; border: none; white-space: pre-wrap; min-height: 150px; max-height: 300px; overflow-y: auto;">{$mensagem_html}</div>
            HTML;

            $modal_footer = <<<HTML
            <button type="button" class="btn btn-primary" onclick="copiarMensagem('{$element_id}')"><i class="fas fa-copy"></i> Copiar</button>
            <a href="https://wa.me/?text={$mensagem_url}" target="_blank" class="btn btn-success ms-2"><i class="fab fa-whatsapp"></i> Enviar</a>
            HTML;
            // --- FIM DA CORREÇÃO ---
            
            return ['modal_header_class' => "modal-header bg-info text-white", 'modal_titulo' => "Informacoes de " . htmlspecialchars($usuario), 'modal_body' => $modal_body, 'modal_footer' => $modal_footer];
        }
        return ['title' => 'Erro!', 'msg' => 'Teste nao encontrado.', 'icon' => 'error'];
    } catch (PDOException $e) {
        return ['modal_titulo' => 'Erro de BD', 'modal_body' => 'Nao foi possivel buscar as informacoes: ' . $e->getMessage()];
    }
}

/**
 * Adicionado seletor de Bouquets (Pacotes)
 */
function adicionar_testes() {
    $conexao = conectar_bd();
    $stmt_bouquets = $conexao->prepare("SELECT id, bouquet_name FROM bouquets ORDER BY bouquet_name ASC");
    $stmt_bouquets->execute();
    $bouquetOptions = "<option value='' selected disabled>Selecione um pacote</option>";
    while ($lista_bouquet = $stmt_bouquets->fetch()) {
        $bouquetOptions .= '<option value="' . $lista_bouquet['id'] . '">' . htmlspecialchars($lista_bouquet['bouquet_name']) . '</option>';
    }

    $modal_body = '<form id="formAdicionarTeste"><input type="hidden" name="confirme_adicionar_testes" value="1">';
    $modal_body .= '<div class="form-group mb-2"><label>Nome:</label><input type="text" class="form-control" name="name" required></div>';
    $modal_body .= '<div class="row mb-2"><div class="form-group col-md-6"><label>Usuario:</label><input type="text" class="form-control" name="usuario" required></div><div class="form-group col-md-6"><label>Senha:</label><input type="text" class="form-control" name="senha" required></div></div>';
    
    $modal_body .= '<div class="row mb-2">';
    $modal_body .= '  <div class="form-group col-md-6"><label>Validade (Horas):</label><input type="number" class="form-control" name="horas" value="4" min="1" required></div>';
    $modal_body .= '  <div class="form-group col-md-6"><label>Pacote (Conteudo):</label><select class="form-select" name="bouquet_id" required>' . $bouquetOptions . '</select></div>';
    $modal_body .= '</div>';

    $modal_body .= '</form>';
    $modal_footer = "<button type='button' class='btn btn-secondary' data-bs-dismiss='modal'>Cancelar</button><button type='button' onclick='enviarFormulario(\"formAdicionarTeste\", \"api/testes.php\")' class='btn btn-info'>Salvar</button>";
    return ['modal_titulo' => "Adicionar Novo Teste", 'modal_body' => $modal_body, 'modal_footer' => $modal_footer];
}

/**
 * Adicionado 'bouquet_id' para ser salvo
 */
function confirme_adicionar_testes($post_data) {
    $conexao = conectar_bd();
    $admin_id = $_SESSION['admin_id'];
    $name = trim($post_data["name"] ?? ''); $usuario = trim($post_data["usuario"] ?? ''); $senha = trim($post_data["senha"] ?? '');
    $horas = (int)($post_data["horas"] ?? 4);
    
    $bouquet_id = (int)($post_data["bouquet_id"] ?? 0);
    $plano_default = ''; // Mantido como vazio
    
    if (empty($name) || empty($usuario) || empty($senha) || empty($bouquet_id)) {
        return ['title' => 'Erro!', 'msg' => 'Todos os campos sao obrigatorios (incluindo Pacote).', 'icon' => 'error'];
    }
    $data_vencimento = date("Y-m-d H:i:s", strtotime("+$horas hours"));
    try {
        $sql = "INSERT INTO clientes (admin_id, name, usuario, senha, Vencimento, is_trial, Criado_em, plano, bouquet_id) VALUES (?, ?, ?, ?, ?, 1, NOW(), ?, ?)";
        $conexao->prepare($sql)->execute([$admin_id, $name, $usuario, $senha, $data_vencimento, $plano_default, $bouquet_id]);
        
        $template = file_exists($_SERVER['DOCUMENT_ROOT'] . '/template_mensagem.txt') ? file_get_contents($_SERVER['DOCUMENT_ROOT'] . '/template_mensagem.txt') : "ERRO: template nao encontrado";
        
        $portal_url = obter_url_load_balance_ou_padrao($conexao);
        
        $pretty_m3u_ts = $portal_url . '/m3u-ts/' . $usuario . '/' . $senha;
        $pretty_m3u_hls = $portal_url . '/m3u-m3u8/' . $usuario . '/' . $senha;
        $pretty_ssiptv = $portal_url . '/ss-ts/' . $usuario . '/' . $senha;

        $replacements = [
            '#username#' => $usuario, '#password#' => $senha, '#exp_date#' => date('d/m/Y H:i', strtotime($data_vencimento)),
            '#url#' => $portal_url,
            '#m3u_link#' => $portal_url . '/get.php?username=' . $usuario . '&password=' . $senha . '&type=m3u_plus&output=ts',
            '#m3u_link_hls#' => $portal_url . '/get.php?username=' . $usuario . '&password=' . $senha . '&type=m3u_plus&output=m3u8',
            '#ssiptv_link#' => $portal_url . '/get.php?username=' . $usuario . '&password=' . $senha . '&type=m3u_plus&output=ts',
            '#m3u_encurtado#' => $pretty_m3u_ts,
            '#m3u_hls_encurtado#' => $pretty_m3u_hls,
            '#ssiptv_encurtado#' => $pretty_ssiptv,
        ];
        $mensagem_final = str_replace(array_keys($replacements), array_values($replacements), $template);
        return ['title' => 'Sucesso!', 'msg' => 'Teste adicionado!', 'icon' => 'success', 'data_table' => 'atualizar', 'dados_cliente_msg' => $mensagem_final];
    } catch (Exception $e) {
        return ['title' => 'Erro!', 'msg' => 'Falha ao adicionar teste: ' . $e->getMessage(), 'icon' => 'error'];
    }
}

function converter_teste($id) {
    try {
        $conexao = conectar_bd();
        $admin_id = $_SESSION['admin_id'];
        $stmt = $conexao->prepare("SELECT usuario FROM clientes WHERE id = ? AND admin_id = ? AND is_trial = 1");
        $stmt->execute([$id, $admin_id]);
        $cliente = $stmt->fetch();
        if(!$cliente) return ['modal_titulo' => 'Erro', 'modal_body' => 'Conta de teste nao encontrada.'];
        
        $modal_body = "<form id='formConverterTeste-{$id}'><input type='hidden' name='confirme_converter_teste' value='$id'></form>";
        $modal_body .= "Deseja converter o teste do usuario <strong>(" . htmlspecialchars($cliente['usuario']) . ")</strong> em um cliente ativo?<br><br>";
        $modal_body .= "Esta acao consumira <strong>1 credito</strong> e definira o vencimento para daqui a <strong>1 mes</strong>.";
        $modal_footer = "<button type='button' class='btn btn-secondary' data-bs-dismiss='modal'>Cancelar</button><button type='button' class='btn btn-success' onclick='enviarFormulario(\"formConverterTeste-{$id}\", \"api/testes.php\")'>Confirmar Conversao</button>";
        return ['modal_header_class' => "modal-header bg-success text-white", 'modal_titulo' => "Converter Teste em Cliente", 'modal_body' => $modal_body, 'modal_footer' => $modal_footer];
    } catch (PDOException $e) {
        return ['modal_titulo' => 'Erro de BD', 'modal_body' => '<div class="alert alert-danger">Erro ao carregar dados: ' . $e->getMessage() . '</div>'];
    }
}

function confirme_converter_teste($post_data) {
    $id = $post_data['confirme_converter_teste'] ?? 0;
    if (empty($id)) return ['title' => "Erro!", 'msg' => "ID do teste invalido.", 'icon' => "error"];

    $conexao = conectar_bd();
    $admin_id = $_SESSION['admin_id'];
    $nova_data = date("Y-m-d 23:59:59", strtotime("+1 month"));
    
    try {
        $stmt_admin = $conexao->prepare("SELECT admin, creditos FROM admin WHERE id = ?");
        $stmt_admin->execute([$admin_id]);
        $admin_info = $stmt_admin->fetch(PDO::FETCH_ASSOC);

        if ( $admin_info['admin'] != 1 && $admin_info['creditos'] < 1) {
            return ['title' => 'Creditos Insuficientes!', 'msg' => 'Voce precisa de pelo menos 1 credito para converter um teste.', 'icon' => 'warning'];
        }

        $conexao->beginTransaction();
        $stmt_update = $conexao->prepare("UPDATE clientes SET Vencimento = ?, is_trial = 0, Ultimo_pagamento = NOW() WHERE id = ? AND admin_id = ?");
        $stmt_update->execute([$nova_data, $id, $admin_id]);
        if ($admin_info['admin'] != 1) {
            $conexao->prepare("UPDATE admin SET creditos = creditos - 1 WHERE id = ?")->execute([$admin_id]);
        }
        $conexao->commit();
        
        return ['title' => "Sucesso!", 'msg' => "Teste convertido em cliente! Redirecionando...", 'icon' => "success", 'data_table' => 'atualizar', 'redirect_url' => 'clientes.php'];
    } catch (Exception $e) {
        if ($conexao->inTransaction()) {
            $conexao->rollBack();
        }
        return ['title' => "Erro!", 'msg' => "Nao foi possivel converter o teste: " . $e->getMessage(), 'icon' => "error"];
    }
}

function delete_teste($id) {
    try {
        $conexao = conectar_bd();
        $admin_id = $_SESSION['admin_id'];
        $stmt = $conexao->prepare("SELECT usuario FROM clientes WHERE id = ? AND admin_id = ?");
        $stmt->execute([$id, $admin_id]);
        $cliente = $stmt->fetch();
        if(!$cliente) return ['modal_titulo' => 'Erro', 'modal_body' => 'Teste nao encontrado ou voce nao tem permissao.'];

        $modal_body = "<form id='formDeletarTeste-{$id}'><input type='hidden' name='confirme_delete_teste' value='{$id}'></form>";
        $modal_body .= "Tem certeza que deseja excluir o teste de <strong>" . htmlspecialchars($cliente['usuario']) . "</strong> permanentemente?";
        $modal_footer = "<button type='button' class='btn btn-secondary' data-bs-dismiss='modal'>Cancelar</button><button type='button' class='btn btn-danger' onclick='enviarFormulario(\"formDeletarTeste-{$id}\", \"api/testes.php\")'>EXCLUIR</button>";
        return ['modal_header_class' => "modal-header bg-danger text-white", 'modal_titulo' => "Excluir Teste", 'modal_body' => $modal_body, 'modal_footer' => $modal_footer];
    } catch (PDOException $e) {
        return ['modal_titulo' => 'Erro de BD', 'modal_body' => '<div class="alert alert-danger">Erro ao carregar dados: ' . $e->getMessage() . '</div>'];
    }
}

function confirme_delete_teste($post_data) {
    $id = $post_data['confirme_delete_teste'] ?? 0;
    if(empty($id)) return ['title' => 'Erro!', 'msg' => 'ID de teste invalido.', 'icon' => 'error'];

    try {
        $conexao = conectar_bd();
        $admin_id = $_SESSION['admin_id'];
        $stmt = $conexao->prepare("DELETE FROM clientes WHERE id = ? AND admin_id = ? AND is_trial = 1"); 
        $stmt->execute([$id, $admin_id]);
        if ($stmt->rowCount() > 0) {
            return ['title' => 'Sucesso!', 'msg' => 'Teste excluido com sucesso!', 'icon' => 'success', 'data_table' => 'atualizar'];
        } else {
            return ['title' => 'Atencao!', 'msg' => 'Nenhum teste foi excluido (ID nao encontrado ou sem permissao).', 'icon' => 'warning'];
        }
    } catch (Exception $e) {
        return ['title' => 'Erro!', 'msg' => 'Falha ao excluir teste: ' . $e->getMessage(), 'icon' => 'error'];
    }
}

/**
 * GERA TESTE RÁPIDO E ABRE O MODAL
 */
function gerar_teste_rapido() {
    $conexao = conectar_bd();
    $admin_id = $_SESSION['admin_id'] ?? null;
    if (empty($admin_id)) {
        return [
            'modal_header_class' => "modal-header bg-danger text-white",
            'modal_titulo'       => "Erro de Sessão",
            'modal_body'         => '<div class="alert alert-danger mb-0">Sessão inválida. Por favor, faça login novamente.</div>',
            'modal_footer'       => '<button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>'
        ];
    }

    $duracao_teste_horas = 4;
    $plano_default = '';
    $bouquet_id_default = 1; // ID 1 = Pacote Completo (padrão)
    
    try {
        $usuario_teste = "teste_" . strtolower(substr(str_shuffle("abcdefghijklmnopqrstuvwxyz0123456789"), 0, 6));
        $senha_teste = strtolower(substr(str_shuffle("abcdefghijklmnopqrstuvwxyz0123456789"), 0, 6));
        $data_vencimento = date('Y-m-d H:i:s', strtotime("+$duracao_teste_horas hours"));

        $sql = "INSERT INTO clientes (admin_id, name, usuario, senha, Vencimento, is_trial, Criado_em, plano, bouquet_id) VALUES (?, ?, ?, ?, ?, 1, NOW(), ?, ?)";
        $conexao->prepare($sql)->execute([$admin_id, 'Teste Rapido', $usuario_teste, $senha_teste, $data_vencimento, $plano_default, $bouquet_id_default]);
        
        $template = file_exists($_SERVER['DOCUMENT_ROOT'] . '/template_mensagem.txt') ? file_get_contents($_SERVER['DOCUMENT_ROOT'] . '/template_mensagem.txt') : "ERRO: template nao encontrado";
        $portal_url = obter_url_load_balance_ou_padrao($conexao);
        
        $pretty_m3u_ts = $portal_url . '/m3u-ts/' . $usuario_teste . '/' . $senha_teste;
        $pretty_m3u_hls = $portal_url . '/m3u-m3u8/' . $usuario_teste . '/' . $senha_teste;
        $pretty_ssiptv = $portal_url . '/ss-ts/' . $usuario_teste . '/' . $senha_teste;

        $replacements = [
            '#username#'         => $usuario_teste,
            '#password#'         => $senha_teste,
            '#url#'              => $portal_url,
            '#exp_date#'         => date('d/m/Y H:i', strtotime($data_vencimento)),
            '#m3u_link#'         => $portal_url . '/get.php?username=' . $usuario_teste . '&password=' . $senha_teste . '&type=m3u_plus&output=ts',
            '#m3u_link_hls#'     => $portal_url . '/get.php?username=' . $usuario_teste . '&password=' . $senha_teste . '&type=m3u_plus&output=m3u8',
            '#ssiptv_link#'      => $portal_url . '/get.php?username=' . $usuario_teste . '&password=' . $senha_teste . '&type=m3u_plus&output=ts',
            '#m3u_encurtado#'    => $pretty_m3u_ts,
            '#m3u_hls_encurtado#' => $pretty_m3u_hls,
            '#ssiptv_encurtado#' => $pretty_ssiptv,
        ];
        $mensagem_final = str_replace(array_keys($replacements), array_values($replacements), $template);

        // --- INÍCIO DA CORREÇÃO ---
        // Variáveis preparadas para o Heredoc
        $mensagem_copy = htmlspecialchars($mensagem_final, ENT_QUOTES);
        $mensagem_html = htmlspecialchars($mensagem_final);
        $mensagem_url = urlencode($mensagem_final);

        $modal_body = <<<HTML
        <div id="infoTesteRapido"
             data-copy-text="{$mensagem_copy}"
             data-raw-text="{$mensagem_copy}"
             class="form-control" readonly style="font-family: monospace; font-size: 0.9rem; resize: none; border: none; white-space: pre-wrap; min-height: 150px; max-height: 300px; overflow-y: auto;">{$mensagem_html}</div>
        HTML;
        
        $modal_footer = <<<HTML
        <button type="button" class="btn btn-primary" onclick="copiarMensagem('infoTesteRapido')"><i class="fas fa-copy"></i> Copiar</button>
        <a href="https://wa.me/?text={$mensagem_url}" target="_blank" class="btn btn-success ms-2"><i class="fab fa-whatsapp"></i> Enviar</a>
        HTML;
        // --- FIM DA CORREÇÃO ---
        
        // Retorna o MODAL, como o 'modal_master' espera
        return [
            'modal_header_class' => "modal-header bg-info text-white",
            'modal_titulo'       => "Teste Rápido Criado: " . htmlspecialchars($usuario_teste),
            'modal_body'         => $modal_body,
            'modal_footer'       => $modal_footer,
            'data_table'         => 'atualizar' // Manda o JS atualizar a tabela de testes
        ];

    } catch (Exception $e) {
        // --- INÍCIO DA CORREÇÃO (Catch) ---
        $error_message = htmlspecialchars($e->getMessage());
        $modal_body_catch = <<<HTML
        <div class="alert alert-danger mb-0">{$error_message}</div>
        HTML;
        // --- FIM DA CORREÇÃO (Catch) ---

        // Se der erro, retorna um modal de erro
        return [
            'modal_header_class' => "modal-header bg-danger text-white",
            'modal_titulo'       => "Erro ao Gerar Teste",
            'modal_body'         => $modal_body_catch, // Variável corrigida
            'modal_footer'       => '<button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Fechar</button>'
        ];
    }
}
?>