<?php
// api/ip_manager.php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
header('Content-Type: application/json'); // Resposta sempre JSON

// --- INÍCIO DA SEGURANÇA ---
try {
    require_once 'controles/db.php';
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Erro crítico: Não foi possível carregar o arquivo do banco de dados.']);
    exit;
}

$conn = conectar_bd(); // Conecta ao BD
if (!$conn) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Falha na conexão com o banco de dados.']);
    exit;
}

// Pega o token do corpo JSON (que o frontend envia)
$data = json_decode(file_get_contents('php://input'), true);
$token_enviado = $data['token'] ?? ''; // O token está DENTRO do JSON

if (empty($token_enviado)) {
    http_response_code(403);
    echo json_encode(['status' => 'error', 'message' => 'Acesso negado: Token de autenticação não fornecido.']);
    exit;
}

try {
    // Valida o token (igual ao api/personalizar.php)
    $sql_token = "SELECT id FROM admin WHERE token = :token";
    $stmt_token = $conn->prepare($sql_token);
    $stmt_token->bindParam(':token', $token_enviado, PDO::PARAM_STR);
    $stmt_token->execute();
    $admin_logado = $stmt_token->fetch(PDO::FETCH_ASSOC);

    if (!$admin_logado || empty($admin_logado['id'])) {
        http_response_code(403);
        echo json_encode(['status' => 'error', 'message' => 'Acesso negado: Token inválido ou expirado.']);
        exit;
    }
    
    // Valida o Nível do Admin (igual ao api/personalizar.php)
    if (!isset($_SESSION['nivel_admin']) || $_SESSION['nivel_admin'] != 1) {
        http_response_code(403);
        echo json_encode(['status' => 'error', 'message' => 'Acesso negado: Apenas o Administrador principal pode gerir IPs.']);
        exit;
    }
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Erro no banco de dados durante a validação do token: ' . $e->getMessage()]);
    exit;
}
// --- FIM DA SEGURANÇA ---


// Processa as ações
$action = $_GET['action'] ?? null;
// $data já foi lido no bloco de segurança

try {
    switch ($action) {
        case 'get_lists':
            // [CORREÇÃO] A consulta agora pega bans expirados (temporários) OU bans permanentes (NULL)
            $stmt_banned = $conn->prepare("SELECT ip_address, reason, ban_timestamp 
                                           FROM banned_ips 
                                           WHERE ban_expires > NOW() OR ban_expires IS NULL 
                                           ORDER BY ban_timestamp DESC");
            $stmt_banned->execute();
            $banned_list = $stmt_banned->fetchAll(PDO::FETCH_ASSOC);

            $stmt_allowed = $conn->prepare("SELECT ip_address, notes FROM allowed_ips ORDER BY id DESC");
            $stmt_allowed->execute();
            $allowed_list = $stmt_allowed->fetchAll(PDO::FETCH_ASSOC);
            
            echo json_encode(['banned' => $banned_list, 'allowed' => $allowed_list]);
            break;

        case 'unblock_ips':
            $ips = $data['ips'] ?? [];
            if (!empty($ips)) {
                $placeholders = implode(',', array_fill(0, count($ips), '?'));
                $stmt = $conn->prepare("DELETE FROM banned_ips WHERE ip_address IN ($placeholders)");
                $stmt->execute($ips);
                echo json_encode(['status' => 'success', 'message' => count($ips) . ' IP(s) desbloqueado(s).']);
            }
            break;

        case 'manual_block_ip':
            $ip = $data['ip'] ?? null;
            $reason = $data['reason'] ?? 'Bloqueio Manual (Admin)';

            if (empty($ip)) {
                echo json_encode(['status' => 'error', 'message' => 'Endereço IP não fornecido.']);
                exit;
            }

            // [CORREÇÃO] Salva o ban permanente com 'ban_expires' como NULL
            $sql = "INSERT INTO banned_ips (ip_address, reason, ban_timestamp, ban_expires) 
                    VALUES (?, ?, NOW(), NULL)
                    ON DUPLICATE KEY UPDATE 
                        reason = VALUES(reason), 
                        ban_timestamp = NOW(),
                        ban_expires = NULL";
            
            $stmt = $conn->prepare($sql);
            $stmt->execute([$ip, $reason]); 

            echo json_encode(['status' => 'success', 'message' => "IP $ip bloqueado com sucesso!"]);
            break;

        case 'allow_ip':
            $ip = $data['ip'] ?? null;
            $notes = $data['notes'] ?? '';
            if ($ip) {
                $stmt = $conn->prepare("INSERT INTO allowed_ips (ip_address, notes) VALUES (?, ?) ON DUPLICATE KEY UPDATE notes = VALUES(notes)");
                $stmt->execute([$ip, $notes]);
                echo json_encode(['status' => 'success', 'message' => 'IP adicionado à whitelist.']);
            }
            break;

        case 'remove_allowed':
            $ip = $data['ip'] ?? null;
            if ($ip) {
                $stmt = $conn->prepare("DELETE FROM allowed_ips WHERE ip_address = ?");
                $stmt->execute([$ip]);
                echo json_encode(['status' => 'success', 'message' => 'IP removido da whitelist.']);
            }
            break;
            
        default:
            echo json_encode(['status' => 'error', 'message' => 'Ação inválida.']);
            break;
    }
} catch (PDOException $e) {
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Erro na base de dados.', 'details' => $e->getMessage()]);
}
?>