<?php
// Arquivo: /api/processar_lista_m3u.php
// [VERSÃO RÁPIDA - SEM TMDB]
// Este script é otimizado para velocidade máxima e processa 15.000 itens de uma vez.
// Ele NÃO busca sinopses ou capas no TMDB.

ini_set('display_errors', 0);
error_reporting(0);
set_time_limit(0); 
ini_set('memory_limit', '1024M'); 

// --- DEPENDÊNCIAS ---
require_once(__DIR__ . '/controles/db.php');
session_start();

// --- FUNÇÃO DE AJUDA: Enviar Erro JSON ---
function sendError($message, $http_code = 400) {
    http_response_code($http_code); 
    echo json_encode(['success' => false, 'message' => $message]);
    exit;
}

// --- VERIFICAÇÃO DE SEGURANÇA (TOKEN) ---
$conexao = conectar_bd();
if (!$conexao) {
    sendError('Falha ao conectar ao banco de dados.', 500);
}
$conexao->exec("SET NAMES 'utf8mb4'");

$token_enviado = $_POST['token'] ?? '';
if (empty($token_enviado)) {
    sendError('Acesso negado: Token de autenticação não fornecido.', 403);
}

try {
    $sql_token = "SELECT id FROM admin WHERE token = :token";
    $stmt_token = $conexao->prepare($sql_token);
    $stmt_token->bindParam(':token', $token_enviado, PDO::PARAM_STR);
    $stmt_token->execute();
    $admin_logado = $stmt_token->fetch(PDO::FETCH_ASSOC);

    if (!$admin_logado || empty($admin_logado['id'])) {
        sendError('Acesso negado: Token inválido ou expirado.', 403);
    }
    $_SESSION['admin_id'] = $admin_logado['id'];
} catch (PDOException $e) {
    sendError('Erro no banco de dados durante a validação do token.', 500);
}

// ===================================================================
// FUNÇÕES DE IMPORTAÇÃO (RÁPIDAS - SEM TMDB)
// ===================================================================

function executarConsulta($conexao, $query, $parametros) {
    try {
        $stmt = $conexao->prepare($query);
        $stmt->execute($parametros);
        return $stmt;
    } catch (PDOException $e) {
        throw new Exception("Erro de SQL: " . $e->getMessage() . " (Query: $query)");
    }
}

// Esta é a versão RÁPIDA. Ela não busca TMDB.
function adicionarLinks_Rapido($links, $conexao) {
    
    $admin_id = $_SESSION['admin_id'] ?? 1; 
    $results = ['urls' => 0, 'movie' => 0, 'live' => 0, 'series' => 0, 'episodios' => 0, 'temporadas' => 0, 'exists' => 0, 'error' => [], 'importedItems' => []];
    $urlsParaVerificar = array_column($links, 'url');
    if (empty($urlsParaVerificar)) return $results;

    $placeholders = implode(',', array_fill(0, count($urlsParaVerificar), '?'));
    $queryLinks = "(SELECT link FROM streams WHERE link IN ($placeholders)) UNION (SELECT link FROM series_episodes WHERE link IN ($placeholders))";
    $stmtLinks = executarConsulta($conexao, $queryLinks, array_merge($urlsParaVerificar, $urlsParaVerificar));
    $linksExistentes = array_flip($stmtLinks->fetchAll(PDO::FETCH_COLUMN));

    $streamsParaInserir = []; $episodiosParaInserir = []; $temporadasParaInserir = []; $seriesParaInserir = [];
    static $categoriasCache = []; static $seriesCache = []; static $seasonsCache = [];
    
    foreach ($links as $link) {
        $url = trim($link['url'] ?? '');
        if (isset($linksExistentes[$url]) || empty($url)) { $results['exists']++; continue; }
        
        $groupTitle = trim($link['category'] ?? 'Sem Categoria');
        $channelName = trim($link['name'] ?? 'Sem Nome');
        $tvgLogo = trim($link['tvgLogo'] ?? '');
        $container = preg_match('/\.(mp4|mkv|ts|m3u8)$/i', $url, $ext) ? strtolower($ext[1]) : 'ts';
        $type = $link['type'] ?? 'movie'; 

        $cacheKeyCat = $groupTitle . '_' . $type;
        if (!isset($categoriasCache[$cacheKeyCat])) {
            $stmtCat = executarConsulta($conexao, "SELECT id FROM categoria WHERE nome = ? AND type = ?", [$groupTitle, $type]);
            $categoria = $stmtCat->fetch(PDO::FETCH_ASSOC);
            if ($categoria) { $categoriasCache[$cacheKeyCat] = $categoria['id']; } 
            else {
                executarConsulta($conexao, "INSERT INTO categoria (nome, type, admin_id) VALUES (?, ?, ?)", [$groupTitle, $type, $admin_id]);
                $categoriasCache[$cacheKeyCat] = $conexao->lastInsertId();
            }
        }
        $category_id = $categoriasCache[$cacheKeyCat];
        
        if ($type === 'series' && preg_match('/^(.*?)(?:[\s._-]+)?(?:[Ss]|Temporada)[\s._-]?(\d{1,2})[\s._-]?(?:[EeXx]|Episodio|x)[\s._-]?(\d{1,3})/i', $channelName, $matches)) {
            $nomeSerie = trim($matches[1]); $temporada = (int)$matches[2]; $episodioNum = (int)$matches[3]; $series_id = null;
            $cacheKeySerie = $nomeSerie . '_' . $category_id;
            if (!isset($seriesCache[$cacheKeySerie])) {
                $stmtSerie = executarConsulta($conexao, "SELECT id FROM series WHERE name = ? AND category_id = ?", [$nomeSerie, $category_id]);
                $serie = $stmtSerie->fetch(PDO::FETCH_ASSOC);
                if ($serie) { $series_id = $serie['id']; $seriesCache[$cacheKeySerie] = $series_id; } 
                else {
                    // Inserção simples (4 colunas)
                    executarConsulta($conexao, "INSERT INTO series (name, category_id, cover, release_date) VALUES (?, ?, ?, ?)", [$nomeSerie, $category_id, $tvgLogo, date('Y-m-d H:i:s')]);
                    $series_id = $conexao->lastInsertId();
                    $seriesCache[$cacheKeySerie] = $series_id;
                    $results['series']++; $results['importedItems'][] = $nomeSerie; 
                }
            } else { $series_id = $seriesCache[$cacheKeySerie]; }
            $cacheKeySeason = $series_id . '_' . $temporada;
            if (!isset($seasonsCache[$cacheKeySeason])) {
                $temporadasParaInserir[] = [$series_id, $temporada, 'Temporada ' . $temporada];
                $seasonsCache[$cacheKeySeason] = true; $results['temporadas']++;
            }
            // Array simples de episódio (10 colunas)
            $episodiosParaInserir[] = [$url, $series_id, $temporada, $episodioNum, $channelName, $container, $tvgLogo, $category_id, null, null];
            $results['episodios']++;
        
        } elseif ($type === 'live' || $type === 'movie' || $type === 'series') {
            // Array simples de stream (17 colunas, a maioria NULL)
            $streamsParaInserir[] = [
                $url, $category_id, $channelName, $type, $container, 
                $tvgLogo, // stream_icon
                null, null, null, null, null, null, null, null, null, null // 10 campos TMDB
            ];
            
            if ($type === 'live') $results['live']++;
            else { $results['movie']++; $results['importedItems'][] = $channelName; }
        }
        $results['urls']++;
    }

    // --- EXECUÇÃO DOS INSERTS ---

    if (!empty($seriesParaInserir)) {
        // (link3.png) - 12 colunas (NÃO tem admin_id)
        $sqlSerie = "INSERT INTO series (name, category_id, cover, release_date) VALUES (?, ?, ?, ?)";
        $stmtSerie = $conexao->prepare($sqlSerie);
        foreach ($seriesParaInserir as $serieData) {
            // Remove os 8 campos TMDB nulos
            $serieDataSimples = array_slice($serieData, 0, 4); 
            $stmtSerie->execute($serieDataSimples); 
            $real_id = $conexao->lastInsertId();
            $temp_id = "temp_" . (array_search($serieData, $seriesParaInserir) + 1);
            $cacheKeySerie = $serieData[0] . '_' . $serieData[1];
            $seriesCache[$cacheKeySerie] = $real_id; 
            foreach ($temporadasParaInserir as $k => $tempData) { if ($tempData[0] == $temp_id) $temporadasParaInserir[$k][0] = $real_id; }
            foreach ($episodiosParaInserir as $k => $epiData) { if ($epiData[1] == $temp_id) $episodiosParaInserir[$k][1] = $real_id; }
        }
    }
    if (!empty($streamsParaInserir)) {
        // Query para STREAMS (sem TMDB)
        $sqlStream = "INSERT INTO streams (link, category_id, name, stream_type, container_extension, stream_icon, added) 
                       VALUES (?, ?, ?, ?, ?, ?, NOW())";
                       
        $stmtStream = $conexao->prepare($sqlStream);
        foreach ($streamsParaInserir as $streamData) { 
             // Pega apenas os 6 primeiros campos (ignora os 10 NULLs do TMDB)
            $streamDataSimples = array_slice($streamData, 0, 6);
            $stmtStream->execute($streamDataSimples); 
        }
    }
    if (!empty($temporadasParaInserir)) {
        $sqlSeason = "INSERT IGNORE INTO series_seasons (series_id, season_number, name) VALUES (?, ?, ?)";
        $stmtSeason = $conexao->prepare($sqlSeason);
        foreach ($temporadasParaInserir as $seasonData) { $stmtSeason->execute($seasonData); }
    }
    if (!empty($episodiosParaInserir)) {
        $sqlEpisode = "INSERT IGNORE INTO series_episodes (link, series_id, season, episode_num, title, container_extension, movie_image, category_id, plot, tmdb_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmtEpisode = $conexao->prepare($sqlEpisode);
        foreach ($episodiosParaInserir as $episodeData) { $stmtEpisode->execute($episodeData); }
    }
    
    return $results;
}

// ===================================================================
// ROTEADOR DE AÇÕES
// ===================================================================

$session_id = $_POST['session_id'] ?? null;
$is_batch_process = isset($_POST['process_batch']);

// --- AÇÃO 1: INICIAR UPLOAD (Primeira chamada) ---
if (!$is_batch_process) {
    $content = '';
    $m3u_file = $_FILES['m3u_file'] ?? null;
    $m3u_url = $_POST['m3u_url'] ?? null;

    try {
        if ($m3u_file && $m3u_file['error'] === UPLOAD_ERR_OK) {
            $content = file_get_contents($m3u_file['tmp_name']);
        } elseif (!empty($m3u_url)) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $m3u_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_TIMEOUT, 60);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.110 Safari/537.36');
            $content = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curl_error = curl_error($ch);
            curl_close($ch);
            if ($curl_error) throw new Exception("Falha de cURL: $curl_error");
            if ($http_code != 200) throw new Exception("O servidor da URL respondeu com um erro: $http_code (Ex: 404 Not Found).");
        } else {
            throw new Exception('Nenhum arquivo ou URL foi enviado.');
        }
        if (empty($content)) { throw new Exception('O arquivo M3U está vazio.'); }

        $content = preg_replace('/^\xEF\xBB\xBF/', '', $content); 
        $content = preg_replace('/^\s+/', '', $content); 

        if (strpos($content, '#EXTM3U') !== 0) {
            throw new Exception('Arquivo M3U inválido. O conteúdo não começa com #EXTM3U.');
        }

        $lines = explode("\n", $content); $content = null; $items = []; $currentData = [];

        foreach ($lines as $line) {
            $line = trim($line); if (empty($line)) continue;
            if (strpos($line, '#EXTINF:') === 0) {
                $currentData = [];
                $nameMatch = preg_match('/,(.*)$/', $line, $matches);
                $currentData['name'] = $nameMatch ? trim($matches[1]) : 'Sem Nome';
                preg_match_all('/([a-zA-Z0-9_-]+)="([^"]*)"/', $line, $attrMatches, PREG_SET_ORDER);
                foreach ($attrMatches as $attr) {
                    if ($attr[1] === 'tvg-logo') $currentData['tvgLogo'] = $attr[2];
                    if ($attr[1] === 'group-title') $currentData['category'] = $attr[2];
                }
            } elseif (strpos($line, 'http') === 0 || strpos($line, 'rtmp') === 0) {
                if (!empty($currentData)) {
                    $currentData['url'] = $line;
                    $category = $currentData['category'] ?? ''; $name = $currentData['name'];
                    if (preg_match('/[Ss]\d{1,2}[\s._-]?[EeXx\s._-]?\d{1,3}/i', $name) || preg_match('/serie|série/i', $category)) { $currentData['type'] = 'series'; } 
                    else if (preg_match('/filme|movie/i', $category)) { $currentData['type'] = 'movie'; } 
                    else { $currentData['type'] = 'live'; }
                    $items[] = $currentData; $currentData = [];
                }
            }
        }
        $lines = null; 

        if (empty($items)) { throw new Exception('Nenhum item válido (http/rtmp) foi encontrado na lista.'); }

        $session_id = uniqid('m3u_', true);
        $backup_dir = __DIR__ . '/../backups/';
        if (!is_dir($backup_dir)) { @mkdir($backup_dir, 0755, true); }
        $temp_file_path = $backup_dir . 'sess_' . $session_id . '.json';
        
        if (file_put_contents($temp_file_path, json_encode($items)) === false) {
            throw new Exception('Falha ao criar arquivo de sessão temporário. Verifique as permissões da pasta /backups/.');
        }

        echo json_encode([
            'success' => true,
            'session_id' => $session_id,
            'total' => count($items),
            'import_mode' => $_POST['import_mode'] ?? 1
        ]);

    } catch (Throwable $t) { sendError($t->getMessage()); }
    exit;
}

// --- AÇÃO 2: PROCESSAR LOTE (Chamadas seguintes) ---
if ($is_batch_process) {
    
    // Pega o batch size que o JS enviou (que será 15000 no modo Extremo)
    $batch_size = (int)($_POST['batch_size'] ?? 1000); 
    
    $page = (int)($_POST['page'] ?? 0);
    $import_mode = (int)($_POST['import_mode'] ?? 1); 
    $useTmdb = false; // TMDB está DESATIVADO

    $temp_file_path = __DIR__ . '/../backups/sess_' . basename($session_id) . '.json';

    try {
        if (!file_exists($temp_file_path)) { throw new Exception("Sessão de importação ($session_id) não encontrada."); }

        $all_items = json_decode(file_get_contents($temp_file_path), true);
        if ($all_items === null) { throw new Exception("Falha ao ler o arquivo de sessão."); }

        $total_items = count($all_items);
        $offset = $page * $batch_size;
        $batch_items = array_slice($all_items, $offset, $batch_size);

        if (empty($batch_items)) {
            @unlink($temp_file_path); 
            echo json_encode(['success' => true, 'is_complete' => true, 'details' => ['processed' => 0, 'details' => []]]);
            exit;
        }

        $conexao->beginTransaction();
        // [CORREÇÃO] Chama a função simples (sem $apiKey)
        $results = adicionarLinks_Rapido($batch_items, $conexao);
        $conexao->commit();

        $logs = [];
        $logs[] = ['tipo' => 'Filmes', 'nome' => $results['movie'], 'mensagem' => 'Filmes processados'];
        $logs[] = ['tipo' => 'Séries', 'nome' => $results['series'], 'mensagem' => 'Séries processadas'];
        $logs[] = ['tipo' => 'Canais', 'nome' => $results['live'], 'mensagem' => 'Canais processados'];
        if ($results['exists'] > 0) { $logs[] = ['tipo' => 'Aviso', 'nome' => $results['exists'], 'mensagem' => 'Itens já existentes (duplicados)']; }
        
        echo json_encode([
            'success' => true,
            'is_complete' => false,
            'details' => [
                'processed' => count($batch_items),
                'details' => $logs
            ]
        ]);

    } catch (Throwable $t) { 
        if ($conexao->inTransaction()) { $conexao->rollBack(); }
        sendError($t->getMessage());
    }
    exit;
}
?>