<?php
// Arquivo: /api/processar_m3u.php
// [VERSÃO DEFINITIVA]
// Este arquivo é 100% standalone e contém todas as funções necessárias,
// sem "emprestar" código de outros importadores.

ini_set('display_errors', 0);
error_reporting(0);
set_time_limit(0); 
ini_set('memory_limit', '1024M'); 

// --- DEPENDÊNCIAS ---
require_once(__DIR__ . '/controles/db.php');
session_start();

// ===================================================================
// FUNÇÃO DE AJUDA: Enviar Erro JSON
// ===================================================================
function sendError($message, $http_code = 400) {
    http_response_code($http_code); 
    echo json_encode(['success' => false, 'message' => $message]);
    exit;
}

// ===================================================================
// VERIFICAÇÃO DE SEGURANÇA (TOKEN)
// ===================================================================
$conexao = conectar_bd();
if (!$conexao) {
    sendError('Falha ao conectar ao banco de dados.', 500);
}
// Força o UTF-8 na conexão
$conexao->exec("SET NAMES 'utf8mb4'");

$token_enviado = $_POST['token'] ?? '';
if (empty($token_enviado)) {
    sendError('Acesso negado: Token de autenticação não fornecido.', 403); // 403 Forbidden
}

try {
    $sql_token = "SELECT id FROM admin WHERE token = :token";
    $stmt_token = $conexao->prepare($sql_token);
    $stmt_token->bindParam(':token', $token_enviado, PDO::PARAM_STR);
    $stmt_token->execute();
    $admin_logado = $stmt_token->fetch(PDO::FETCH_ASSOC);

    if (!$admin_logado || empty($admin_logado['id'])) {
        sendError('Acesso negado: Token inválido ou expirado.', 403); // 403 Forbidden
    }
    // Define o admin_id na sessão (necessário para a função 'adicionarLinks_Otimizado')
    $_SESSION['admin_id'] = $admin_logado['id'];

} catch (PDOException $e) {
    sendError('Erro no banco de dados durante a validação do token.', 500);
}

// ===================================================================
// FUNÇÕES DE IMPORTAÇÃO (Copiadas do importer2026)
// ===================================================================

$tmdb_api_key = "f99aa9ae1fe7619969cc7db0938c1ae5"; // API Key (igual ao outro)
$tmdb_logs = []; // Log global para o TMDB

function executarConsulta($conexao, $query, $parametros) {
    $stmt = $conexao->prepare($query);
    $stmt->execute($parametros);
    return $stmt;
}

function getTmdbInfo($title, $type, $apiKey) {
    global $tmdb_logs; 
    if (empty($apiKey) || $apiKey == "SUA_API_KEY_AQUI") {
        $tmdb_logs[] = "[AVISO] TMDB pulado: API Key não configurada.";
        return null;
    }
    $year = null;
    $cleanTitle = $title; 
    if (preg_match('/(?:[\s._-]|–|—)?\s*\((\d{4})\)\s*$/', $title, $matches)) {
        $year = $matches[1];
        $cleanTitle = trim(preg_replace('/(?:[\s._-]|–|—)?\s*\((\d{4})\)\s*$/', '', $title));
    }
    $searchType = ($type == 'movie') ? 'movie' : 'tv';
    $url = "https://api.themoviedb.org/3/search/{$searchType}?api_key={$apiKey}&query=" . urlencode($cleanTitle) . "&language=pt-BR";
    if ($year) $url .= "&year=" . $year;

    $ch = curl_init($url);
    curl_setopt_array($ch, [CURLOPT_RETURNTRANSFER => true, CURLOPT_TIMEOUT => 10, CURLOPT_SSL_VERIFYPEER => false]);
    $response = curl_exec($ch);
    $curl_error = curl_error($ch); 
    curl_close($ch);
    if ($curl_error) { $tmdb_logs[] = "[ERRO cURL Search] '{$cleanTitle}': {$curl_error}"; return null; }

    $data = json_decode($response, true);
    if (isset($data['results']) && !empty($data['results'])) {
        $result = $data['results'][0]; 
        return [
            'tmdb_id' => $result['id'] ?? null,
            'plot' => $result['overview'] ?? null,
            'cover' => $result['poster_path'] ? "https://image.tmdb.org/t/p/w500" . $result['poster_path'] : null,
            'cover_big' => $result['poster_path'] ? "https://image.tmdb.org/t/p/original" . $result['poster_path'] : null,
            'backdrop_path' => $result['backdrop_path'] ? "https://image.tmdb.org/t/p/w1280" . $result['backdrop_path'] : null,
            'release_date' => $result['release_date'] ?? $result['first_air_date'] ?? null,
            'rating' => $result['vote_average'] ?? null,
            'genre' => null, 'director' => null, 'cast' => null, 'youtube_trailer' => null,
        ];
    }
    $tmdb_logs[] = "[INFO] TMDB: Nenhum resultado para '{$cleanTitle}' (Ano: {$year}).";
    return null;
}

function adicionarLinks_Otimizado($links, $conexao, $useTmdb, $apiKey) {
    global $tmdb_logs; 
    $tmdb_logs = []; 
    $admin_id = $_SESSION['admin_id'] ?? 1; 
    $results = ['urls' => 0, 'movie' => 0, 'live' => 0, 'series' => 0, 'episodios' => 0, 'temporadas' => 0, 'exists' => 0, 'error' => [], 'importedItems' => []];
    $urlsParaVerificar = array_column($links, 'url');
    if (empty($urlsParaVerificar)) return $results;

    $placeholders = implode(',', array_fill(0, count($urlsParaVerificar), '?'));
    $queryLinks = "(SELECT link FROM streams WHERE link IN ($placeholders)) UNION (SELECT link FROM series_episodes WHERE link IN ($placeholders))";
    $stmtLinks = executarConsulta($conexao, $queryLinks, array_merge($urlsParaVerificar, $urlsParaVerificar));
    $linksExistentes = array_flip($stmtLinks->fetchAll(PDO::FETCH_COLUMN));

    $streamsParaInserir = []; $episodiosParaInserir = []; $temporadasParaInserir = []; $seriesParaInserir = [];
    static $categoriasCache = []; static $seriesCache = []; static $seasonsCache = [];

    foreach ($links as $link) {
        $url = trim($link['url'] ?? '');
        if (isset($linksExistentes[$url]) || empty($url)) { $results['exists']++; continue; }
        $groupTitle = trim($link['category'] ?? 'Sem Categoria');
        $channelName = trim($link['name'] ?? 'Sem Nome');
        $tvgLogo = trim($link['tvgLogo'] ?? '');
        $container = preg_match('/\.(mp4|mkv|ts|m3u8)$/i', $url, $ext) ? strtolower($ext[1]) : 'ts';
        $type = $link['type'] ?? 'movie'; 
        
        $cacheKeyCat = $groupTitle . '_' . $type;
        if (!isset($categoriasCache[$cacheKeyCat])) {
            $stmtCat = executarConsulta($conexao, "SELECT id FROM categoria WHERE nome = ? AND type = ?", [$groupTitle, $type]);
            $categoria = $stmtCat->fetch(PDO::FETCH_ASSOC);
            if ($categoria) { $categoriasCache[$cacheKeyCat] = $categoria['id']; } 
            else {
                executarConsulta($conexao, "INSERT INTO categoria (nome, type, admin_id) VALUES (?, ?, ?)", [$groupTitle, $type, $admin_id]);
                $categoriasCache[$cacheKeyCat] = $conexao->lastInsertId();
            }
        }
        $category_id = $categoriasCache[$cacheKeyCat];

        if ($type === 'series' && preg_match('/^(.*?)(?:[\s._-]+)?(?:[Ss]|Temporada)[\s._-]?(\d{1,2})[\s._-]?(?:[EeXx]|Episodio|x)[\s._-]?(\d{1,3})/i', $channelName, $matches)) {
            $nomeSerie = trim($matches[1]); $temporada = (int)$matches[2]; $episodioNum = (int)$matches[3]; $series_id = null;
            $cacheKeySerie = $nomeSerie . '_' . $category_id;
            if (!isset($seriesCache[$cacheKeySerie])) {
                $stmtSerie = executarConsulta($conexao, "SELECT id FROM series WHERE name = ? AND category_id = ?", [$nomeSerie, $category_id]);
                $serie = $stmtSerie->fetch(PDO::FETCH_ASSOC);
                if ($serie) { $series_id = $serie['id']; $seriesCache[$cacheKeySerie] = ['id' => $series_id]; } 
                else {
                    $tmdbData = $useTmdb ? getTmdbInfo($nomeSerie, 'series', $apiKey) : null;
                    $release_date_final = $tmdbData['release_date'] ?? date('Y-m-d H:i:s');
                    $seriesData = [$nomeSerie, $category_id, $tmdbData['cover'] ?? $tvgLogo, $tmdbData['plot'] ?? null, $tmdbData['cast'] ?? null, $tmdbData['director'] ?? null, $tmdbData['genre'] ?? null, $release_date_final, $tmdbData['rating'] ?? null, $tmdbData['backdrop_path'] ?? null, $tmdbData['youtube_trailer'] ?? null, $tmdbData['tmdb_id'] ?? null];
                    $seriesParaInserir[] = $seriesData; $series_id = "temp_" . count($seriesParaInserir); $seriesCache[$cacheKeySerie] = ['id' => $series_id];
                    $results['series']++; $results['importedItems'][] = $nomeSerie; 
                }
            } else { $series_id = $seriesCache[$cacheKeySerie]['id']; }
            $cacheKeySeason = $series_id . '_' . $temporada;
            if (!isset($seasonsCache[$cacheKeySeason])) {
                $temporadasParaInserir[] = [$series_id, $temporada, 'Temporada ' . $temporada];
                $seasonsCache[$cacheKeySeason] = true; $results['temporadas']++;
            }
            $episodiosParaInserir[] = [$url, $series_id, $temporada, $episodioNum, $channelName, $container, $tvgLogo, $category_id, null, null];
            $results['episodios']++;
        } elseif ($type === 'live') {
            $streamsParaInserir[] = [$url, $category_id, $channelName, $type, $container, $tvgLogo, null, null, null, null, null, null, null, null, null, null];
            $results[$type]++;
        } elseif ($type === 'movie' || $type === 'series') {
            $tmdbData = $useTmdb ? getTmdbInfo($channelName, 'movie', $apiKey) : null;
            $streamsParaInserir[] = [$url, $category_id, $channelName, 'movie', $container, $tmdbData['cover'] ?? $tvgLogo, $tmdbData['tmdb_id'] ?? null, $tmdbData['cover_big'] ?? null, $tmdbData['plot'] ?? null, $tmdbData['cast'] ?? null, $tmdbData['director'] ?? null, $tmdbData['genre'] ?? null, $tmdbData['release_date'] ?? null, $tmdbData['rating'] ?? null, $tmdbData['youtube_trailer'] ?? null, $tmdbData['backdrop_path'] ?? null];
            $results['movie']++; $results['importedItems'][] = $channelName; 
        }
        $results['urls']++;
    }

    if (!empty($seriesParaInserir)) {
        $sqlSerie = "INSERT INTO series (name, category_id, cover, plot, cast, director, genre, release_date, rating, backdrop_path, youtube_trailer, tmdb_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmtSerie = $conexao->prepare($sqlSerie);
        foreach ($seriesParaInserir as $serieData) {
            $stmtSerie->execute($serieData); $real_id = $conexao->lastInsertId();
            $temp_id = "temp_" . (array_search($serieData, $seriesParaInserir) + 1);
            $cacheKeySerie = $serieData[0] . '_' . $serieData[1];
            $seriesCache[$cacheKeySerie]['id'] = $real_id; 
            foreach ($temporadasParaInserir as $k => $tempData) { if ($tempData[0] == $temp_id) $temporadasParaInserir[$k][0] = $real_id; }
            foreach ($episodiosParaInserir as $k => $epiData) { if ($epiData[1] == $temp_id) $episodiosParaInserir[$k][1] = $real_id; }
        }
    }
    if (!empty($streamsParaInserir)) {
        $sqlStream = "INSERT INTO streams (link, category_id, name, stream_type, container_extension, stream_icon, added, tmdb_id, cover_big, plot, cast, director, genre, release_date, rating, youtube_trailer, backdrop_path) VALUES (?, ?, ?, ?, ?, ?, NOW(), ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmtStream = $conexao->prepare($sqlStream);
        foreach ($streamsParaInserir as $streamData) { $stmtStream->execute($streamData); }
    }
    if (!empty($temporadasParaInserir)) {
        $sqlSeason = "INSERT IGNORE INTO series_seasons (series_id, season_number, name) VALUES (?, ?, ?)";
        $stmtSeason = $conexao->prepare($sqlSeason);
        foreach ($temporadasParaInserir as $seasonData) { $stmtSeason->execute($seasonData); }
    }
    if (!empty($episodiosParaInserir)) {
        $sqlEpisode = "INSERT IGNORE INTO series_episodes (link, series_id, season, episode_num, title, container_extension, movie_image, category_id, plot, tmdb_id) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
        $stmtEpisode = $conexao->prepare($sqlEpisode);
        foreach ($episodiosParaInserir as $episodeData) { $stmtEpisode->execute($episodeData); }
    }
    $results['error'] = array_merge($results['error'], $tmdb_logs); 
    return $results;
}

// ===================================================================
// ROTEADOR DE AÇÕES
// ===================================================================

$session_id = $_POST['session_id'] ?? null;
$is_batch_process = isset($_POST['process_batch']);

// --- AÇÃO 1: INICIAR UPLOAD (Primeira chamada) ---
if (!$is_batch_process) {
    $content = '';
    $m3u_file = $_FILES['m3u_file'] ?? null;
    $m3u_url = $_POST['m3u_url'] ?? null;

    try {
        if ($m3u_file && $m3u_file['error'] === UPLOAD_ERR_OK) {
            $content = file_get_contents($m3u_file['tmp_name']);
        } elseif (!empty($m3u_url)) {
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $m3u_url);
            curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
            curl_setopt($ch, CURLOPT_TIMEOUT, 60);
            curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
            curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/58.0.3029.110 Safari/537.36');
            $content = curl_exec($ch);
            $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            $curl_error = curl_error($ch);
            curl_close($ch);
            if ($curl_error) throw new Exception("Falha de cURL: $curl_error");
            if ($http_code != 200) throw new Exception("O servidor da URL respondeu com um erro: $http_code (Ex: 404 Not Found).");
        } else {
            throw new Exception('Nenhum arquivo ou URL foi enviado.');
        }

        if (empty($content)) {
            throw new Exception('O arquivo M3U está vazio.');
        }

        $content = preg_replace('/^\xEF\xBB\xBF/', '', $content); 
        $content = preg_replace('/^\s+/', '', $content); 

        if (strpos($content, '#EXTM3U') !== 0) {
            throw new Exception('Arquivo M3U inválido. O conteúdo não começa com #EXTM3U.');
        }

        $lines = explode("\n", $content);
        $content = null; 
        $items = [];
        $currentData = [];

        foreach ($lines as $line) {
            $line = trim($line);
            if (empty($line)) continue;

            if (strpos($line, '#EXTINF:') === 0) {
                $currentData = [];
                $nameMatch = preg_match('/,(.*)$/', $line, $matches);
                $currentData['name'] = $nameMatch ? trim($matches[1]) : 'Sem Nome';
                preg_match_all('/([a-zA-Z0-9_-]+)="([^"]*)"/', $line, $attrMatches, PREG_SET_ORDER);
                foreach ($attrMatches as $attr) {
                    if ($attr[1] === 'tvg-logo') $currentData['tvgLogo'] = $attr[2];
                    if ($attr[1] === 'group-title') $currentData['category'] = $attr[2];
                }
            } elseif (strpos($line, 'http') === 0 || strpos($line, 'rtmp') === 0) {
                if (!empty($currentData)) {
                    $currentData['url'] = $line;
                    $category = $currentData['category'] ?? '';
                    $name = $currentData['name'];
                    if (preg_match('/[Ss]\d{1,2}[\s._-]?[EeXx\s._-]?\d{1,3}/i', $name) || preg_match('/serie|série/i', $category)) {
                        $currentData['type'] = 'series';
                    } else if (preg_match('/filme|movie/i', $category)) {
                        $currentData['type'] = 'movie';
                    } else {
                        $currentData['type'] = 'live';
                    }
                    $items[] = $currentData;
                    $currentData = [];
                }
            }
        }
        $lines = null; 

        if (empty($items)) {
            throw new Exception('Nenhum item válido (http/rtmp) foi encontrado na lista.');
        }

        $session_id = uniqid('m3u_', true);
        $temp_file_path = __DIR__ . '/../backups/sess_' . $session_id . '.json';
        
        if (file_put_contents($temp_file_path, json_encode($items)) === false) {
            throw new Exception('Falha ao criar arquivo de sessão temporário. Verifique as permissões da pasta /backups/.');
        }

        echo json_encode([
            'success' => true,
            'session_id' => $session_id,
            'total' => count($items),
            'import_mode' => $_POST['import_mode'] ?? 1
        ]);

    } catch (Exception $e) {
        sendError($e->getMessage());
    }
    exit;
}

// --- AÇÃO 2: PROCESSAR LOTE (Chamadas seguintes) ---
if ($is_batch_process) {
    
    $page = (int)($_POST['page'] ?? 0);
    $batch_size = (int)($_POST['batch_size'] ?? 500);
    $import_mode = (int)($_POST['import_mode'] ?? 1); 
    $useTmdb = true; 

    $temp_file_path = __DIR__ . '/../backups/sess_' . basename($session_id) . '.json';

    try {
        if (!file_exists($temp_file_path)) {
            throw new Exception("Sessão de importação ($session_id) não encontrada ou expirada.");
        }

        $all_items = json_decode(file_get_contents($temp_file_path), true);
        if ($all_items === null) {
            throw new Exception("Falha ao ler o arquivo de sessão.");
        }

        $total_items = count($all_items);
        $offset = $page * $batch_size;
        $batch_items = array_slice($all_items, $offset, $batch_size);

        if (empty($batch_items)) {
            @unlink($temp_file_path); 
            echo json_encode(['success' => true, 'is_complete' => true, 'details' => ['processed' => 0, 'details' => []]]);
            exit;
        }

        $conexao->beginTransaction();
        $results = adicionarLinks_Otimizado($batch_items, $conexao, $useTmdb, $tmdb_api_key);
        $conexao->commit();

        $logs = [];
        $logs[] = ['tipo' => 'Filmes', 'nome' => $results['movie'], 'mensagem' => 'Filmes processados'];
        $logs[] = ['tipo' => 'Séries', 'nome' => $results['series'], 'mensagem' => 'Séries processadas'];
        $logs[] = ['tipo' => 'Canais', 'nome' => $results['live'], 'mensagem' => 'Canais processados'];
        if ($results['exists'] > 0) {
            $logs[] = ['tipo' => 'Aviso', 'nome' => $results['exists'], 'mensagem' => 'Itens já existentes (duplicados)'];
        }
        foreach ($results['error'] as $errMsg) {
             $logs[] = ['tipo' => 'Erro', 'nome' => 'TMDB', 'mensagem' => $errMsg];
        }

        echo json_encode([
            'success' => true,
            'is_complete' => false,
            'details' => [
                'processed' => count($batch_items),
                'details' => $logs
            ]
        ]);

    } catch (Exception $e) {
        if ($conexao->inTransaction()) {
            $conexao->rollBack();
        }
        sendError($e->getMessage());
    }
    exit;
}
?>