<?php
header('Content-Type: application/json; charset=utf-8');
require_once __DIR__ . '/../controles/db.php';

// Apenas aceita requisições do tipo POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405); // Method Not Allowed
    die(json_encode(['status' => 'error', 'message' => 'Método não permitido.']));
}

$data = json_decode(file_get_contents("php://input"), true);

// Validação dos dados de entrada
if (!isset($data['id']) || !is_numeric($data['id']) || !isset($data['meses']) || !is_numeric($data['meses'])) {
    http_response_code(400); // Bad Request
    die(json_encode(['status' => 'error', 'message' => 'ID do cliente ou número de meses inválido.']));
}

$clienteId = $data['id'];
$mesesParaAdicionar = (int)$data['meses'];

try {
    $pdo = conectar_bd();
    
    // 1. Primeiro, busca o cliente para pegar a data de vencimento atual
    $stmt = $pdo->prepare("SELECT * FROM clientes WHERE id = :id");
    $stmt->execute([':id' => $clienteId]);
    $cliente = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$cliente) {
        http_response_code(404);
        die(json_encode(['status' => 'error', 'message' => 'Cliente não encontrado.']));
    }

    // 2. Calcula a nova data de vencimento de forma inteligente
    $agora = new DateTime("now", new DateTimeZone('America/Sao_Paulo'));
    $dataBase = new DateTime($cliente['Vencimento']);

    // Se a data de vencimento já passou, renova a partir de HOJE.
    // Se não, renova a partir da DATA DE VENCIMENTO FUTURA.
    if ($dataBase < $agora) {
        $dataBase = $agora;
    }
    
    $dataBase->add(new DateInterval("P{$mesesParaAdicionar}M")); // Adiciona o(s) mês(es)
    $novoVencimento = $dataBase->format('Y-m-d H:i:s');

    // 3. Atualiza o cliente no banco de dados com a nova data
    // Também garante que ele não seja mais um 'teste' (is_trial = 0)
    $updateStmt = $pdo->prepare(
        "UPDATE clientes SET Vencimento = :vencimento, is_trial = 0 WHERE id = :id"
    );
    $updateStmt->execute([
        ':vencimento' => $novoVencimento,
        ':id' => $clienteId
    ]);

    // 4. Prepara os dados para retornar ao JavaScript
    $dadosRetorno = [
        'usuario' => $cliente['usuario'],
        'senha' => $cliente['senha'],
        'Vencimento' => $novoVencimento // Envia a nova data formatada
    ];

    // 5. Envia a resposta de sucesso COM os dados necessários
    echo json_encode(['status' => 'success', 'data' => $dadosRetorno]);

} catch (Exception $e) {
    http_response_code(500);
    die(json_encode(['status' => 'error', 'message' => 'Erro de servidor ao tentar renovar o cliente: ' . $e->getMessage()]));
}
?>