<?php
require_once 'menu.php';

// Inicia a sessao para pegar o ID do admin
// Coloque isso no topo, a menos que seu painel ja inicie a sessao em outro arquivo.
if (session_status() == PHP_SESSION_NONE) {
    session_start();
}

// --- 1. CONEXAO E VARIAVEIS ---

// Voce precisa incluir seu arquivo de conexao principal aqui
// CAMINHO CORRIGIDO de acordo com sua solicitacao
require_once 'api/controles/db.php'; // Ou onde quer que esteja sua funcao conectar_bd()

// Pega o ID do admin logado
if (!isset($_SESSION['admin_id'])) {
    die("Erro: Voce precisa estar logado para acessar esta pagina.");
}
$admin_id = (int)$_SESSION['admin_id'];

$mensagem_sucesso = '';
$mensagem_erro = '';
$conexao = null; // Inicia a variavel

try {
    $conexao = conectar_bd(); // Usa sua funcao de conexao

    // --- 2. LOGICA PARA SALVAR (UPDATE) ---
    if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['salvar_configuracoes'])) {
        
        $dias = (int)($_POST['confianca_dias'] ?? 0);
        $ativo = (int)($_POST['confianca_ativo'] ?? 0);

        // Garante que 'ativo' seja apenas 0 ou 1
        $ativo = ($ativo == 1) ? 1 : 0;
        // Garante que dias nao seja negativo
        $dias = ($dias < 0) ? 0 : $dias;

        $stmt = $conexao->prepare("UPDATE admin SET confianca_dias = ?, confianca_ativo = ? WHERE id = ?");
        
        if ($stmt->execute([$dias, $ativo, $admin_id])) {
            $mensagem_sucesso = "Configuracoes salvas com sucesso!";
        } else {
            $mensagem_erro = "Erro ao salvar as configuracoes.";
        }
    }

    // --- 3. LOGICA PARA BUSCAR DADOS (SELECT) ---
    $stmt_select = $conexao->prepare("SELECT confianca_dias, confianca_ativo FROM admin WHERE id = ?");
    $stmt_select->execute([$admin_id]);
    $config_atual = $stmt_select->fetch(PDO::FETCH_ASSOC);

    // Se nao encontrar, define valores padrao
    if (!$config_atual) {
        $config_atual = ['confianca_dias' => 0, 'confianca_ativo' => 0];
        // Voce pode querer registrar um erro aqui se o admin nao for encontrado
    }

} catch (PDOException $e) {
    $mensagem_erro = "Erro de conexao com o banco: " . $e->getMessage();
}

// Fechar a conexao (opcional, dependendo da sua funcao conectar_bd())
$conexao = null;

?>

<!DOCTYPE html>
<html lang="pt-br">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Configuracao de Confianca</title>
    
    <!-- 
      IMPORTANTE: Voce deve incluir aqui os mesmos links de CSS e JS
      que o seu painel de admin usa (Bootstrap, FontAwesome, etc.)
    -->
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <!-- Adicionando FontAwesome (necessario para os icones) -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/5.15.4/css/all.min.css">
    
    <link rel="stylesheet" href="assets/css/style.css"> <!-- O CSS do seu painel (pode ser sobrescrito abaixo) -->

    <style>
        /* --- NOVO DESIGN PROFISSIONAL - TEMA ESCURO --- */
        
        :root {
            --dark-bg-primary: #1e1e2d; /* Fundo principal escuro */
            --dark-bg-secondary: #27293d; /* Fundo do cartao */
            --dark-border: #3a3b50;      /* Cor da borda */
            --dark-text-primary: #e0e0e0; /* Cor do texto principal */
            --dark-text-secondary: #a0a0b9;/* Cor do texto secundario (muted) */
            --dark-green: #28a745;
            --dark-red: #dc3545;
            --dark-blue-gradient: linear-gradient(45deg, #007bff, #0056b3);
        }

        body {
            background-color: var(--dark-bg-primary);
            color: var(--dark-text-primary);
        }

        .config-card {
            background: var(--dark-bg-secondary);
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.2);
            border: 1px solid var(--dark-border);
            margin-top: 30px;
            overflow: hidden;
        }

        .config-header {
            background: var(--dark-blue-gradient);
            color: #fff;
            padding: 1.5rem 2rem;
        }
        
        .config-header h3 {
            margin-bottom: 0.25rem;
            font-weight: 600;
        }

        .config-header p {
            margin-bottom: 0;
            opacity: 0.8;
            font-size: 0.95rem;
        }

        .config-body {
            padding: 2rem;
        }
        
        /* Ajuste nos Alertas para tema escuro */
        .alert-success {
            background-color: rgba(40, 167, 69, 0.1);
            border-color: rgba(40, 167, 69, 0.5);
            color: #90ee90; /* Verde claro */
        }
        .alert-danger {
            background-color: rgba(220, 53, 69, 0.1);
            border-color: rgba(220, 53, 69, 0.5);
            color: #f08080; /* Vermelho claro */
        }

        /* Estilo para os Radio Cards (Ativo/Inativo) */
        .status-options {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }
        
        .form-label {
            color: var(--dark-text-primary);
        }
        
        .form-text {
            color: var(--dark-text-secondary);
        }

        .status-box {
            border: 2px solid var(--dark-border);
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s ease;
            position: relative;
        }

        .status-box i {
            font-size: 2.5rem;
            display: block;
            margin-bottom: 10px;
        }
        
        .status-box span {
            font-weight: 600;
            font-size: 1.1rem;
            color: var(--dark-text-primary); /* Texto claro */
        }

        .status-options input[type="radio"] {
            display: none;
        }

        /* Cores dos icones */
        .status-box.active i { color: var(--dark-green); }
        .status-box.inactive i { color: var(--dark-red); }

        /* Estilo quando SELECIONADO */
        .status-options input[type="radio"]:checked + .status-box {
            box-shadow: 0 4px 12px rgba(0,0,0,0.1);
            transform: translateY(-2px);
        }
        
        .status-options input[type="radio"]:checked + .status-box.active {
            border-color: var(--dark-green);
            background: rgba(40, 167, 69, 0.1); /* Fundo verde escuro translucido */
        }
        
        .status-options input[type="radio"]:checked + .status-box.inactive {
            border-color: var(--dark-red);
            background: rgba(220, 53, 69, 0.1); /* Fundo vermelho escuro translucido */
        }
        
        /* Input de Dias com Icone */
        .input-group-text {
            background-color: #3a3b50; /* Cinza escuro */
            border: 1px solid var(--dark-border);
            border-right: none;
            color: var(--dark-text-secondary);
        }
        
        .form-control {
            background-color: var(--dark-bg-primary); /* Fundo mais escuro */
            border: 1px solid var(--dark-border);
            color: #fff; /* Texto branco */
        }

        .form-control::placeholder {
            color: var(--dark-text-secondary);
        }

        .form-control:focus {
            background-color: var(--dark-bg-primary);
            color: #fff;
            border-color: #007bff;
            box-shadow: 0 0 0 0.25rem rgba(0,123,255,.25);
        }
        
        .config-footer {
            background-color: var(--dark-bg-secondary);
            padding: 1.5rem 2rem;
            border-top: 1px solid var(--dark-border);
            text-align: right;
        }
        
        .btn-primary {
            font-weight: 600;
            padding: 0.75rem 1.5rem;
            /* Garante que o botao se destaque */
            background-color: #007bff;
            border-color: #007bff;
        }

    </style>
</head>
<body class="dark-mode"> <!-- Classe opcional para o body -->

    <!-- 
      IMPORTANTE: Voce deve incluir aqui o menu/header do seu painel
      para que esta pagina seja exibida "dentro" dele.
      
      Exemplo (apenas ilustrativo):
      <?php // include 'partes/header.php'; ?>
      <?php // include 'partes/menu_lateral.php'; ?>
    -->

    <!-- Conteudo da Pagina -->
    <div class="container-fluid">
        <!-- Grid para centralizar o cartao -->
        <div class="row justify-content-center">
            <div class="col-lg-7 col-md-9">
                
                <form action="" method="POST" class="config-card">
                    <!-- Input escondido para identificar o form -->
                    <input type="hidden" name="salvar_configuracoes" value="1">

                    <!-- CABECALHO DO CARTAO -->
                    <div class="config-header">
                        <h3>Configurar Renovacao de Confianca</h3>
                        <p>Ative e defina os dias de liberacao para seus clientes.</p>
                    </div>

                    <!-- CORPO DO CARTAO -->
                    <div class="config-body">
                        
                        <!-- Alertas de Sucesso ou Erro -->
                        <?php if ($mensagem_sucesso): ?>
                            <div class="alert alert-success">
                                <i class="fas fa-check-circle"></i> <?php echo $mensagem_sucesso; ?>
                            </div>
                        <?php endif; ?>

                        <?php if ($mensagem_erro): ?>
                            <div class="alert alert-danger">
                                <i class="fas fa-exclamation-triangle"></i> <?php echo $mensagem_erro; ?>
                            </div>
                        <?php endif; ?>

                        <!-- 4. O FORMULARIO HTML ATUALIZADO -->

                        <div class="mb-4">
                            <label class="form-label fw-bold">Status da Funcao</label>
                            
                            <div class="status-options">
                                <!-- Opcao ATIVO -->
                                <div>
                                    <input type="radio" id="status_ativo" name="confianca_ativo" value="1" 
                                           <?php if ($config_atual['confianca_ativo'] == 1) echo 'checked'; ?>>
                                    <label for="status_ativo" class="status-box active">
                                        <i class="fas fa-check-circle"></i>
                                        <span>Ativo</span>
                                    </label>
                                </div>
                                
                                <!-- Opcao INATIVO -->
                                <div>
                                    <input type="radio" id="status_inativo" name="confianca_ativo" value="0" 
                                           <?php if ($config_atual['confianca_ativo'] == 0) echo 'checked'; ?>>
                                    <label for="status_inativo" class="status-box inactive">
                                        <i class="fas fa-times-circle"></i>
                                        <span>Inativo</span>
                                    </label>
                                </div>
                            </div>
                            <div class="form-text mt-2">
                                "Ativo" mostra o botao de confianca para o cliente. "Inativo" esconde.
                            </div>
                        </div>
                        
                        <hr class="my-4" style="border-color: var(--dark-border);">
                        
                        <div class="mb-3">
                            <label for="dias" class="form-label fw-bold">Dias de Liberacao</label>
                            <div class="input-group">
                                <span class="input-group-text">
                                    <i class="fas fa-calendar-day fa-fw"></i>
                                </span>
                                <input type="number" id="dias" name="confianca_dias" 
                                       class="form-control" 
                                       value="<?php echo htmlspecialchars($config_atual['confianca_dias']); ?>"
                                       min="0"
                                       placeholder="Ex: 2">
                            </div>
                            <div class="form-text">
                                Quantidade de dias que o cliente recebera (0 para desativar).
                            </div>
                        </div>
                    </div>

                    <!-- RODAPE DO CARTAO -->
                    <div class="config-footer">
                        <button type="submit" class="btn btn-primary">
                            <i class="fas fa-save"></i> Salvar Configuracoes
                        </button>
                    </div>

                </form>

            </div>
        </div>
    </div> <!-- Fim do container do painel -->

    <!-- 
      IMPORTANTE: Voce deve incluir aqui os scripts JS do seu painel
      (Bootstrap, JQuery, etc.)
    -->
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="assets/js/main.js"></script> <!-- O JS do seu painel -->

</body>
</html>

