<?php
// ARQUIVO: /gerenciador/ajax/filmes_excluir_massa.php (VERSÃO CORRIGIDA E SEGURA)

// 1. Definir o cabeçalho para garantir que a resposta seja sempre JSON.
header('Content-Type: application/json');

// 2. Iniciar um array de resposta padrão. Isso ajuda a garantir uma saída consistente.
$response = ['success' => false, 'message' => 'Ocorreu um erro inesperado.'];

// 3. Verificar o método da requisição. Ações destrutivas devem usar POST.
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $response['message'] = 'Método de requisição inválido. Use POST.';
    echo json_encode($response);
    exit;
}

// 4. Validar os dados recebidos (array de IDs).
$ids = $_POST['ids'] ?? null;
if (empty($ids) || !is_array($ids)) {
    $response['message'] = 'Nenhum filme foi selecionado ou o formato dos dados é inválido.';
    echo json_encode($response);
    exit;
}

// 5. Medida de segurança CRÍTICA: Garantir que todos os IDs sejam números inteiros positivos.
// 'intval' converte cada item para inteiro. 'array_filter' remove os que resultarem em 0 ou false.
$ids_para_excluir = array_map('intval', $ids);
$ids_para_excluir = array_filter($ids_para_excluir, function($id) {
    return $id > 0;
});

if (empty($ids_para_excluir)) {
    $response['message'] = 'Nenhum ID válido foi fornecido para a exclusão.';
    echo json_encode($response);
    exit;
}

try {
    // 6. Conectar ao banco de dados.
    // O caminho deve ser relativo a ESTE arquivo (/gerenciador/ajax/).
    require_once('../../api/controles/db.php');
    $pdo = conectar_bd();

    if (!$pdo) {
        // Lança uma exceção se a conexão falhar, que será capturada pelo bloco catch.
        throw new Exception("Falha crítica: Não foi possível conectar ao banco de dados.");
    }

    // 7. Preparar a query SQL de forma segura, usando placeholders.
    // Cria uma string com a quantidade correta de '?' (ex: "?,?,?").
    $placeholders = implode(',', array_fill(0, count($ids_para_excluir), '?'));
    
    // CORREÇÃO: A tabela correta é 'streams', conforme seu arquivo 'filmes.php'.
    $sql = "DELETE FROM streams WHERE id IN ($placeholders)";
    
    $stmt = $pdo->prepare($sql);
    
    // 8. Executar a query passando o array de IDs sanitizados.
    if ($stmt->execute($ids_para_excluir)) {
        $count = $stmt->rowCount(); // Pega o número real de linhas afetadas.
        $response['success'] = true;
        $response['message'] = $count . ' filme(s) excluído(s) com sucesso!';
    } else {
        $response['message'] = 'Ocorreu uma falha ao executar o comando de exclusão.';
    }

} catch (PDOException $e) {
    // Captura erros específicos do banco de dados (ex: tabela não existe, erro de sintaxe).
    // Em produção, é melhor logar o erro detalhado e mostrar uma mensagem genérica.
    // error_log('Erro no banco de dados em filmes_excluir_massa: ' . $e->getMessage());
    $response['message'] = 'Erro no servidor de banco de dados. A operação foi cancelada.';

} catch (Exception $e) {
    // Captura outras exceções (ex: falha na conexão, arquivo db.php não encontrado).
    $response['message'] = 'Erro geral no servidor: ' . $e->getMessage();
}

// 9. Enviar a resposta final em formato JSON para o JavaScript.
echo json_encode($response);

?>