<?php
// =================================================================================
// ENDPOINT DE ESTATÍSTICAS: estatisticas_conexoes.php (FINAL E CORRIGIDO)
// RESPONSÁVEL POR FORNECER OS DADOS DOS CARDS VIA JSON
// =================================================================================

header('Content-Type: application/json');

// 1. INCLUSÕES E VERIFICAÇÕES DE SEGURANÇA
// ---------------------------------------------------------------------------------
$db_path = '../api/controles/db.php';
if (!file_exists($db_path)) {
    http_response_code(500);
    die(json_encode(['error' => 'Arquivo de conexão db.php não encontrado.']));
}
require_once($db_path);

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verifica se o admin está logado
if (!isset($_SESSION['logged_in_fxtream']) || $_SESSION['logged_in_fxtream'] !== true) {
    http_response_code(401);
    die(json_encode(['error' => 'Acesso não autorizado. Sessão expirada.']));
}

$loggedInUserId = $_SESSION['admin_id'] ?? 0;
$userLevel = $_SESSION['nivel_admin'] ?? 0;

try {
    $pdo = conectar_bd();
    if (!$pdo) {
        throw new Exception("Falha ao conectar ao banco de dados.");
    }
} catch (Exception $e) {
    http_response_code(500);
    die(json_encode(['error' => 'Erro de conexão com o BD.']));
}

// Filtro de Revendedor (Usando alias 'c' para clientes)
$reseller_filter_sql = '';
if ($userLevel != 1) {
    // Definimos o filtro baseado na coluna CORRETA: admin_id
    $reseller_filter_sql = " AND c.admin_id = :admin_id ";
}


// ------------------------------------------------
// 2. CONSULTA DE DADOS E FILTROS (CORRIGIDA)
// ------------------------------------------------

// Consulta Unificada: CORRIGIDA com INNER JOIN para garantir o filtro de admin_id
$sql = "
    SELECT 
        COUNT(co.id) AS total_conexoes,
        COUNT(DISTINCT co.usuario) AS usuarios_unicos,
        COUNT(DISTINCT IFNULL(s.name, se.title)) AS streams_unicos
    FROM conexoes co
    INNER JOIN clientes c ON co.usuario = c.usuario /* USO DO INNER JOIN PARA FILTRAR */
    LEFT JOIN streams s ON co.stream_id = s.id
    LEFT JOIN series_episodes se ON s.id IS NULL AND co.stream_id = se.id
    WHERE 1=1 {$reseller_filter_sql}";

try {
    $stmt = $pdo->prepare($sql);

    // CORREÇÃO: bindValue CONDICIONAL
    if ($userLevel != 1) {
        $stmt->bindValue(':admin_id', $loggedInUserId, PDO::PARAM_INT);
    }
    
    $stmt->execute();
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);

    // 3. RETORNO JSON
    // ------------------------------------------------
    echo json_encode([
        'status' => 'success',
        'total_conexoes' => $stats['total_conexoes'] ?? 0,
        'usuarios_unicos' => $stats['usuarios_unicos'] ?? 0,
        'streams_unicos' => $stats['streams_unicos'] ?? 0
    ]);

} catch (PDOException $e) {
    http_response_code(500);
    die(json_encode(['error' => 'Erro na consulta de estatísticas: ' . $e->getMessage()]));
}