<?php
// =================================================================================
// TABELA ISOLADA PARA AJAX: tabela_conexoes.php (FINAL E CORRIGIDO)
// RESPONSÁVEL POR GERAR O HTML DO CORPO DA TABELA EM TEMPO REAL
// =================================================================================

// ------------------------------------------------
// 1. INCLUSÕES E VERIFICAÇÕES DE SEGURANÇA
// ------------------------------------------------

$db_path = '../api/controles/db.php';
if (!file_exists($db_path)) {
    // Retorna um erro visível na tabela
    die("<tr><td colspan='5' class='text-center p-3'>ERRO: Arquivo de conexão db.php não encontrado.</td></tr>");
}
require_once($db_path);

if (session_status() === PHP_SESSION_NONE) {
    session_start();
}

// Verifica se o admin está logado
if (!isset($_SESSION['logged_in_fxtream']) || $_SESSION['logged_in_fxtream'] !== true) {
    die("<tr><td colspan='5' class='text-center p-3'>ERRO: Sessão de administrador não ativa ou expirada.</td></tr>");
}

$loggedInUserId = $_SESSION['admin_id'] ?? 0;
$userLevel = $_SESSION['nivel_admin'] ?? 0;

try {
    $pdo = conectar_bd();
    if (!$pdo) {
        throw new Exception("Falha ao conectar ao banco de dados.");
    }
} catch (Exception $e) {
    die("<tr><td colspan='5' class='text-center p-3'>ERRO: Falha na conexão com o BD.</td></tr>");
}

// Filtro de Revendedor (Usando alias 'c' para clientes)
$reseller_filter_sql = '';
if ($userLevel != 1) {
    // Definimos o filtro baseado na coluna CORRETA: admin_id
    $reseller_filter_sql = " AND c.admin_id = :admin_id ";
}

// ------------------------------------------------
// 2. FUNÇÃO DE TEMPO (MANTIDA)
// ------------------------------------------------

/**
 * Calcula o tempo decorrido desde a última atividade, destacando atividades recentes.
 */
function formatar_tempo_passado($datetime_str) {
    if (empty($datetime_str)) return 'N/A';
    
    date_default_timezone_set('America/Sao_Paulo'); 
    
    try {
        $agora = new DateTime(); 
        $ultima_atividade = new DateTime($datetime_str);
        
        // Se a última atividade foi há menos de 30 segundos, mostramos como 'AGORA'
        if ($ultima_atividade->getTimestamp() > ($agora->getTimestamp() - 30)) {
            return '<strong class="text-success">AGORA</strong>';
        }

        $diff = $agora->diff($ultima_atividade);
        
        if ($diff->days > 0) return 'há ' . $diff->days . 'd';
        if ($diff->h > 0) return 'há ' . $diff->h . 'h';
        if ($diff->i > 0) return 'há ' . $diff->i . ' min';
        
        return 'há ' . $diff->s . ' seg';
    } catch (Exception $e) { 
        return 'Data inválida'; 
    }
}

// ------------------------------------------------
// 3. CONSULTA SQL E GERAÇÃO DO HTML (CORRIGIDA)
// ------------------------------------------------

// Consulta Detalhada CORRIGIDA (com INNER JOIN para garantir o filtro de admin_id)
$sql_detalhado = "
    SELECT 
        co.usuario, co.ip, co.user_agent, co.ultima_atividade,
        IFNULL(s.name, se.title) AS stream_name,
        IFNULL(s.stream_type, 'series') AS stream_type 
    FROM conexoes co
    INNER JOIN clientes c ON co.usuario = c.usuario /* USO DO INNER JOIN PARA FILTRAR */
    LEFT JOIN streams s ON co.stream_id = s.id
    LEFT JOIN series_episodes se ON s.id IS NULL AND co.stream_id = se.id
    WHERE 1=1 {$reseller_filter_sql}
    ORDER BY co.ultima_atividade DESC;";
    
$stmt_detalhado = $pdo->prepare($sql_detalhado);

// CORREÇÃO: bindValue CONDICIONAL
if ($userLevel != 1) {
    $stmt_detalhado->bindValue(':admin_id', $loggedInUserId, PDO::PARAM_INT);
}
$stmt_detalhado->execute();
$conexoes_detalhadas = $stmt_detalhado->fetchAll(PDO::FETCH_ASSOC);

// GERA O HTML DO CORPO DA TABELA
if (empty($conexoes_detalhadas)): ?>
    <tr><td colspan="5" class="text-center p-3">Nenhuma conexão ativa no momento.</td></tr>
<?php else: foreach ($conexoes_detalhadas as $conexao): ?>
    <tr>
        <td><strong><?= htmlspecialchars($conexao['usuario']) ?></strong></td>
        <td>
            <?php
                $tipo = htmlspecialchars($conexao['stream_type'] ?? 'vod');
                $cor_badge = 'secondary';
                if ($tipo === 'live') $cor_badge = 'success';
                if ($tipo === 'movie') $cor_badge = 'info';
            ?>
            <span class="badge bg-<?= $cor_badge ?> me-2"><?= ucfirst($tipo) ?></span>
            <?= htmlspecialchars($conexao['stream_name'] ?? 'Conteúdo VOD') ?>
        </td>
        <td><?= htmlspecialchars($conexao['ip']) ?></td>
        <td><small class="text-muted"><?= formatar_tempo_passado($conexao['ultima_atividade']) ?></small></td>
        <td class="text-center">
            <button class="btn btn-danger btn-sm kill-individual-btn" 
                    data-usuario="<?= htmlspecialchars($conexao['usuario']) ?>" 
                    data-ip="<?= htmlspecialchars($conexao['ip']) ?>" 
                    title="Finalizar Conexão">
                <i class="fas fa-times"></i>
            </button>
        </td>
    </tr>
<?php endforeach; endif; ?>