<?php
// Arquivo: player_api_xui_complete.php
// Versão: v1.0 - API Player compatível com XUI ONE para Android, Smart TV e Roku
// [FIX] Adicionada verificação de 'is_banned' na autenticação do usuário.
// [FIX] Convertido todos os operadores '??' para 'isset()' para compatibilidade com PHP 5.6+.

require_once($_SERVER['DOCUMENT_ROOT'] . '/api/controles/db.php');
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Credentials: true');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization, X-Requested-With');

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    http_response_code(204);
    exit;
}

date_default_timezone_set('America/Sao_Paulo');

// Erros em log mas não expostos ao usuário final
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', $_SERVER['DOCUMENT_ROOT'] . '/php_player_api_xui_error.log');

// Parâmetros (Corrigido para PHP 5.6+)
$username = isset($_GET['username']) ? $_GET['username'] : null;
$password = isset($_GET['password']) ? $_GET['password'] : null;
$action = isset($_GET['action']) ? $_GET['action'] : null;
$category_id_param = isset($_GET['category_id']) ? $_GET['category_id'] : null;
$series_id = isset($_GET['series_id']) ? $_GET['series_id'] : null;
$vod_id = isset($_GET['vod_id']) ? $_GET['vod_id'] : null;

if ($category_id_param === '*' || $category_id_param === 'all') $category_id_param = null;

// Validação básica de usuário e senha
if (!$username || !$password) {
    http_response_code(401);
    error_log("API XUI ERRO: Usuário ou Senha ausentes");
    echo json_encode(['user_info' => ['auth' => 0, 'status' => 'Erro', 'message' => 'Usuário e Senha necessários']]);
    exit();
}

// Conecta ao DB
try {
    $conexao = conectar_bd();
    $conexao->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    http_response_code(500);
    error_log("API XUI - Erro DB: " . $e->getMessage());
    echo json_encode(['user_info' => ['auth' => 0, 'status' => 'Erro', 'message' => 'Erro interno no servidor (DB)']]);
    exit();
}

// Autenticação do usuário
try {
    $query_auth = "SELECT * FROM clientes WHERE usuario = :username AND senha = :password";
    $stmt_auth = $conexao->prepare($query_auth);
    $stmt_auth->bindParam(':username', $username, PDO::PARAM_STR);
    $stmt_auth->bindParam(':password', $password, PDO::PARAM_STR);
    $stmt_auth->execute();
    $user = $stmt_auth->fetch(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    http_response_code(500);
    error_log("API XUI - Erro autenticação: " . $e->getMessage());
    echo json_encode(['user_info' => ['auth' => 0, 'status' => 'Erro', 'message' => 'Erro interno (Auth)']]);
    exit();
}

if (!$user) {
    http_response_code(401);
    error_log("API XUI - Falha autenticação usuário: $username");
    echo json_encode(['user_info' => ['auth' => 0, 'status' => 'Credenciais inválidas']]);
    exit();
}

$bouquet_id = isset($user['bouquet_id']) ? (int)$user['bouquet_id'] : 0;
$adulto = isset($user['adulto']) ? (int)$user['adulto'] : 0;

// =================================================================
// --- INÍCIO DA CORREÇÃO (VERIFICAÇÃO DE BLOQUEIO E EXPIRAÇÃO) ---
// =================================================================
$exp_date_unix = !empty($user['Vencimento']) ? strtotime($user['Vencimento']) : 0;
$is_expired = $exp_date_unix > 0 && $exp_date_unix < time();

// Verifica se a coluna 'is_banned' existe e se é igual a 1
$is_banned = isset($user['is_banned']) && $user['is_banned'] == 1;

// Define o status baseado na prioridade: Banido > Expirado > Ativo
if ($is_banned) {
    $status = 'Banned'; // O app pode ler 'Banned', 'Expired' ou 'Suspended'
} elseif ($is_expired) {
    $status = 'Expired';
} else {
    $status = 'Active';
}

// Se estiver expirado OU banido, a autenticação falha (auth = 0)
$auth = ($is_expired || $is_banned) ? 0 : 1;
// =================================================================
// --- FIM DA CORREÇÃO ---
// =================================================================

// Correção para PHP 5.6+ (removido ??)
$max_connections = max(1, (int)(isset($user['conexoes']) ? $user['conexoes'] : 1));

// Função para formatar streams e séries no padrão XUI ONE
function format_stream_row($row, $num, $type) {
    // Correção para PHP 5.6+ (removido ??)
    $rating = isset($row['rating']) ? $row['rating'] : 0;
    $rating_5based = isset($row['rating_5based']) ? $row['rating_5based'] : round($rating / 2, 1);
    $backdrop = !empty($row['backdrop_path']) ? explode(',', $row['backdrop_path']) : [];

    if ($type === 'live') {
        return [
            'num' => $num,
            'name' => htmlspecialchars_decode(isset($row['name']) ? $row['name'] : ''),
            'stream_type' => 'live',
            'stream_id' => (string)(isset($row['id']) ? $row['id'] : ''),
            'stream_icon' => isset($row['stream_icon']) ? $row['stream_icon'] : '',
            'epg_channel_id' => isset($row['epg_channel_id']) ? $row['epg_channel_id'] : '',
            'added' => isset($row['added']) ? $row['added'] : '',
            'category_id' => (string)(isset($row['category_id']) ? $row['category_id'] : '0'),
            'custom_sid' => isset($row['custom_sid']) ? $row['custom_sid'] : '',
            'tv_archive' => (int)(isset($row['tv_archive']) ? $row['tv_archive'] : 0),
            'tv_archive_duration' => (int)(isset($row['tv_archive_duration']) ? $row['tv_archive_duration'] : 0),
            'is_adult' => (isset($row['is_adult']) && $row['is_adult']) ? '1' : '0',
            'direct_source' => isset($row['direct_source']) ? $row['direct_source'] : ''
        ];
    } elseif ($type === 'movie') {
        return [
            'num' => $num,
            'name' => htmlspecialchars_decode(isset($row['name']) ? $row['name'] : ''),
            'stream_type' => 'movie',
            'stream_id' => (string)(isset($row['id']) ? $row['id'] : ''),
            'stream_icon' => isset($row['stream_icon']) ? $row['stream_icon'] : '',
            'rating' => (string)$rating,
            'rating_5based' => $rating_5based,
            'added' => isset($row['added']) ? $row['added'] : '',
            'category_id' => (string)(isset($row['category_id']) ? $row['category_id'] : '0'),
            'container_extension' => isset($row['container_extension']) ? $row['container_extension'] : 'mp4',
            'custom_sid' => isset($row['custom_sid']) ? $row['custom_sid'] : '',
            'is_adult' => (isset($row['is_adult']) && $row['is_adult']) ? '1' : '0',
            'direct_source' => isset($row['direct_source']) ? $row['direct_source'] : ''
        ];
    } elseif ($type === 'series') {
        return [
            'num' => $num,
            'name' => htmlspecialchars_decode(isset($row['name']) ? $row['name'] : ''),
            'series_id' => (string)(isset($row['id']) ? $row['id'] : ''),
            'cover' => isset($row['cover']) ? $row['cover'] : '',
            'plot' => isset($row['plot']) ? $row['plot'] : '',
            'cast' => isset($row['cast']) ? $row['cast'] : '',
            'director' => isset($row['director']) ? $row['director'] : '',
            'genre' => isset($row['genre']) ? $row['genre'] : '',
            'releaseDate' => isset($row['release_date']) ? $row['release_date'] : '',
            'last_modified' => isset($row['last_modified']) ? $row['last_modified'] : null,
            'rating' => (string)$rating,
            'rating_5based' => $rating_5based,
            'backdrop_path' => $backdrop,
            'youtube_trailer' => isset($row['youtube_trailer']) ? $row['youtube_trailer'] : '',
            'episode_run_time' => isset($row['episode_run_time']) ? $row['episode_run_time'] : '',
            'category_id' => (string)(isset($row['category_id']) ? $row['category_id'] : '0')
        ];
    }
}

// Load balancing servidores
$selected_ip = $_SERVER['SERVER_NAME'];
$selected_port = $_SERVER['SERVER_PORT'];

try {
    $stmt_servers = $conexao->prepare("SELECT server_ip, server_port FROM servers WHERE status = 1");
    $stmt_servers->execute();
    $active_servers = $stmt_servers->fetchAll(PDO::FETCH_ASSOC);
    if (!empty($active_servers)) {
        $sel_srv = $active_servers[array_rand($active_servers)];
        $selected_ip = $sel_srv['server_ip'] ?: $selected_ip;
        $selected_port = $sel_srv['server_port'] ?: $selected_port;
    }
} catch (PDOException $e) {
    error_log("API XUI - Load Balance falhou: " . $e->getMessage());
}

$protocol = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') || $_SERVER['SERVER_PORT'] == 443 ? 'https' : 'http';

// Default action - informações globais de autenticação
if ($action === null) {
    echo json_encode([
        'user_info' => [
            'username' => (string)$user['usuario'],
            'password' => (string)$user['senha'],
            'message' => 'Bem-vindo!',
            'auth' => $auth, // <--- VARIÁVEL CORRIGIDA
            'status' => $status, // <--- VARIÁVEL CORRIGIDA
            'exp_date' => $exp_date_unix > 0 ? (string)$exp_date_unix : null,
            'is_trial' => $user['is_trial'] ? '1' : '0',
            'active_cons' => 0,
            'created_at' => !empty($user['Criado_em']) ? (string)strtotime($user['Criado_em']) : null,
            'max_connections' => (string)$max_connections,
            'allowed_output_formats' => ['m3u8', 'ts']
        ],
        'server_info' => [
            'url' => $selected_ip,
            'port' => (string)$selected_port,
            'https_port' => '443',
            'server_protocol' => $protocol,
            'rtmp_port' => '1935',
            'timestamp_now' => (string)time(),
            'time_now' => date('Y-m-d H:i:s'),
            'timezone' => date_default_timezone_get() ?: 'America/Sao_Paulo'
        ]
    ], JSON_UNESCAPED_SLASHES);
    exit();
}

// Função para montar categorias
function build_categories($type, $bouquet_id, $adulto, $conexao, $username) {
    $params = [':type' => $type];
    if ($bouquet_id == 1) {
        $sql = "SELECT id as category_id, nome as category_name, parent_id FROM categoria WHERE type = :type";
        if ($adulto == 0) $sql .= " AND is_adult = 0";
        $sql .= " ORDER BY position ASC";
    } elseif ($bouquet_id > 1) {
        $sql = "SELECT DISTINCT c.id as category_id, c.nome as category_name, c.parent_id
                 FROM categoria c INNER JOIN bouquet_items bi ON c.id = bi.category_id
                 WHERE c.type = :type AND bi.bouquet_id = :bouquet_id";
        if ($adulto == 0) $sql .= " AND c.is_adult = 0";
        $sql .= " ORDER BY c.position ASC";
        $params[':bouquet_id'] = $bouquet_id;
    } else {
        error_log("API XUI - Categories: Bouquet inválido para $username");
        echo json_encode([]);
        exit();
    }
    try {
        $stmt = $conexao->prepare($sql);
        $stmt->bindParam(':type', $params[':type'], PDO::PARAM_STR);
        if (isset($params[':bouquet_id'])) $stmt->bindParam(':bouquet_id', $params[':bouquet_id'], PDO::PARAM_INT);
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);
        foreach ($results as &$cat) {
            $cat['category_id'] = (string)$cat['category_id'];
            $cat['parent_id'] = isset($cat['parent_id']) ? (string)$cat['parent_id'] : "0";
        }
        return $results;
    } catch (PDOException $e) {
        error_log("API XUI - Erro categories: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Erro interno [Categories]']);
        exit();
    }
}

// Função para montar streams ou séries da tabela conforme filtros
function build_streams_or_series($table, $stream_type, $category_id_param, $bouquet_id, $adulto, $conexao, $username) {
    $params = [':stream_type' => $stream_type];
    $sql = '';
    $category_id_filter = $category_id_param !== null ? (int)$category_id_param : null;

    if ($bouquet_id == 1) {
        $sql = "SELECT t.* FROM `$table` t WHERE t.stream_type = :stream_type";
        if ($category_id_filter !== null) {
            $sql .= " AND t.category_id = :category_id";
            $params[':category_id'] = $category_id_filter;
        }
        if ($adulto == 0) $sql .= " AND t.is_adult = 0";
    } elseif ($bouquet_id > 1) {
        $sql = "SELECT DISTINCT t.* FROM `$table` t INNER JOIN bouquet_items bi ON t.category_id = bi.category_id
                 WHERE t.stream_type = :stream_type AND bi.bouquet_id = :bouquet_id";
        if ($category_id_filter !== null) {
            $sql .= " AND t.category_id = :category_id";
            $params[':category_id'] = $category_id_filter;
        }
        if ($adulto == 0) $sql .= " AND t.is_adult = 0";
        $params[':bouquet_id'] = $bouquet_id;
    } else {
        error_log("API XUI - Bouquet inválido para streams $username");
        echo json_encode([]);
        exit();
    }

    try {
        $stmt = $conexao->prepare($sql);
        $stmt->bindParam(':stream_type', $params[':stream_type'], PDO::PARAM_STR);
        if (isset($params[':category_id'])) $stmt->bindParam(':category_id', $params[':category_id'], PDO::PARAM_INT);
        if (isset($params[':bouquet_id'])) $stmt->bindParam(':bouquet_id', $params[':bouquet_id'], PDO::PARAM_INT);
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (PDOException $e) {
        error_log("API XUI - Erro streams: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Erro interno [Streams]']);
        exit();
    }
}

// Handling categories endpoints
if (in_array($action, ['get_live_categories', 'get_vod_categories', 'get_series_categories'])) {
    $type = str_replace(['get_', '_categories'], '', $action);
    if ($type === 'vod') $type = 'movie';
    $categories = build_categories($type, $bouquet_id, $adulto, $conexao, $username);
    echo json_encode($categories, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit();
}

// Handling streams and series endpoints
if (in_array($action, ['get_live_streams', 'get_vod_streams', 'get_series'])) {
    $stream_type = '';
    $table = '';
    switch ($action) {
        case 'get_live_streams': $stream_type = 'live'; $table = 'streams'; break;
        case 'get_vod_streams': $stream_type = 'movie'; $table = 'streams'; break;
        case 'get_series': $stream_type = 'series'; $table = 'series'; break;
    }
    $results_raw = build_streams_or_series($table, $stream_type, $category_id_param, $bouquet_id, $adulto, $conexao, $username);
    $results_formatted = [];
    $num = 0;
    foreach ($results_raw as $row) {
        $num++;
        $results_formatted[] = format_stream_row($row, $num, $stream_type);
    }
    echo json_encode($results_formatted, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
    exit();
}

// VOD info endpoint
if ($action === 'get_vod_info' && isset($vod_id)) {
    $vod_id_int = filter_var($vod_id, FILTER_VALIDATE_INT);
    if (!$vod_id_int) {
        http_response_code(400);
        echo json_encode(['error' => 'ID de VOD inválido']);
        exit();
    }
    try {
        $query_vod = $conexao->prepare("SELECT * FROM streams WHERE id = :id AND stream_type = 'movie'");
        $query_vod->bindParam(':id', $vod_id_int, PDO::PARAM_INT);
        $query_vod->execute();
        $row = $query_vod->fetch(PDO::FETCH_ASSOC);
        if (!$row) {
            http_response_code(404);
            echo json_encode(['error' => 'VOD Não encontrado']);
            exit();
        }
        function timeToSeconds($timeStr) {
            $parts = explode(':', $timeStr);
            if (count($parts) == 3) return ($parts[0]*3600) + ($parts[1]*60) + $parts[2];
            if (count($parts) == 2) return ($parts[0]*60) + $parts[1];
            return 0;
        }
        // Correção para PHP 5.6+ (removido ??)
        $duration = isset($row['duration']) ? $row['duration'] : '0';
        $durationSecs = timeToSeconds($duration);
        $rating = isset($row['rating']) ? $row['rating'] : 0;
        $rating_5based = isset($row['rating_5based']) ? $row['rating_5based'] : round($rating / 2, 1);
        $backdrop = !empty($row['backdrop_path']) ? explode(',', $row['backdrop_path']) : [];
        $info = [
            'movie_image' => isset($row['stream_icon']) ? $row['stream_icon'] : '',
            'name' => htmlspecialchars_decode(isset($row['name']) ? $row['name'] : ''),
            'plot' => isset($row['plot']) ? $row['plot'] : '',
            'cast' => isset($row['actors']) ? $row['actors'] : (isset($row['cast']) ? $row['cast'] : ''),
            'director' => isset($row['director']) ? $row['director'] : '',
            'genre' => isset($row['genre']) ? $row['genre'] : '',
            'releasedate' => isset($row['releasedate']) ? $row['releasedate'] : '',
            'rating' => (string)$rating,
            'rating_5based' => $rating_5based,
            'youtube_trailer' => isset($row['youtube_trailer']) ? $row['youtube_trailer'] : '',
            'episode_run_time' => $duration,
            'duration_secs' => $durationSecs,
            'duration' => $duration,
            'backdrop_path' => $backdrop
        ];
        $movie_data = [
            'stream_id' => (string)$row['id'],
            'name' => htmlspecialchars_decode(isset($row['name']) ? $row['name'] : ''),
            'stream_type' => 'movie',
            'stream_icon' => isset($row['stream_icon']) ? $row['stream_icon'] : '',
            'added' => isset($row['added']) ? $row['added'] : '',
            'category_id' => (string)(isset($row['category_id']) ? $row['category_id'] : ''),
            'container_extension' => isset($row['container_extension']) ? $row['container_extension'] : 'mp4',
            'custom_sid' => isset($row['custom_sid']) ? $row['custom_sid'] : '',
            'direct_source' => isset($row['direct_source']) ? $row['direct_source'] : ''
        ];
        echo json_encode(['info' => $info, 'movie_data' => $movie_data, 'video' => [], 'audio' => [], 'subtitles' => []], JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        exit();
    } catch (PDOException $e) {
        error_log("API XUI - Erro get_vod_info: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Erro interno [get_vod_info]']);
        exit();
    }
}

// Série info endpoint
if ($action === 'get_series_info' && isset($series_id)) {
    $series_id_int = filter_var($series_id, FILTER_VALIDATE_INT);
    if (!$series_id_int) {
        http_response_code(400);
        echo json_encode(['error' => 'ID de série inválido']);
        exit();
    }
    try {
        // Info série
        $query_series = $conexao->prepare("SELECT * FROM series WHERE id = :id");
        $query_series->bindParam(':id', $series_id_int, PDO::PARAM_INT);
        $query_series->execute();
        $series = $query_series->fetch(PDO::FETCH_ASSOC);
        if (!$series) {
            http_response_code(404);
            echo json_encode(['error' => 'Série não encontrada']);
            exit();
        }
        // Correção para PHP 5.6+ (removido ??)
        $rating = isset($series['rating']) ? $series['rating'] : 0;
        $rating_5based = isset($series['rating_5based']) ? $series['rating_5based'] : round($rating / 2, 1);
        $backdrop = !empty($series['backdrop_path']) ? explode(',', $series['backdrop_path']) : [];
        $info = [
            'name' => htmlspecialchars_decode(isset($series['name']) ? $series['name'] : ''),
            'cover' => isset($series['cover']) ? $series['cover'] : '',
            'plot' => isset($series['plot']) ? $series['plot'] : '',
            'cast' => isset($series['cast']) ? $series['cast'] : '',
            'director' => isset($series['director']) ? $series['director'] : '',
            'genre' => isset($series['genre']) ? $series['genre'] : '',
            'releaseDate' => isset($series['release_date']) ? $series['release_date'] : '',
            'last_modified' => isset($series['last_modified']) ? $series['last_modified'] : null,
            'rating' => (string)$rating,
            'rating_5based' => $rating_5based,
            'backdrop_path' => $backdrop,
            'youtube_trailer' => isset($series['youtube_trailer']) ? $series['youtube_trailer'] : '',
            'episode_run_time' => isset($series['episode_run_time']) ? $series['episode_run_time'] : '',
            'category_id' => (string)(isset($series['category_id']) ? $series['category_id'] : '0')
        ];
        // Episódios agrupados
        $query_eps = $conexao->prepare("SELECT * FROM series_episodes WHERE series_id = :id ORDER BY season ASC, episode_num ASC");
        $query_eps->bindParam(':id', $series_id_int, PDO::PARAM_INT);
        $query_eps->execute();
        $episodes_grouped = [];
        function timeToSecondsEp($str) {
            $parts = explode(':', $str);
            if (count($parts) == 3) return ($parts[0]*3600) + ($parts[1]*60) + $parts[2];
            if (count($parts) == 2) return ($parts[0]*60) + $parts[1];
            return 0;
        }
        while ($ep = $query_eps->fetch(PDO::FETCH_ASSOC)) {
            // Correção para PHP 5.6+ (removido ??)
            $season = isset($ep['season']) ? $ep['season'] : 0;
            $duration_ep = isset($ep['duration']) ? $ep['duration'] : '0';
            $cover_ep = isset($ep['movie_image']) ? $ep['movie_image'] : (isset($ep['cover_big']) ? $ep['cover_big'] : '');

            if (!isset($episodes_grouped[$season])) $episodes_grouped[$season] = [];
            $episodes_grouped[$season][] = [
                'id' => (string)(isset($ep['id']) ? $ep['id'] : ''),
                'episode_num' => isset($ep['episode_num']) ? $ep['episode_num'] : null,
                'title' => htmlspecialchars_decode(isset($ep['title']) ? $ep['title'] : ''),
                'container_extension' => isset($ep['container_extension']) ? $ep['container_extension'] : 'mp4',
                'info' => [
                    'movie_image' => $cover_ep,
                    'plot' => isset($ep['plot']) ? $ep['plot'] : '',
                    'releasedate' => isset($ep['air_date']) ? $ep['air_date'] : '',
                    'rating' => isset($ep['rating']) ? (string)$ep['rating'] : '0',
                    'duration_secs' => timeToSecondsEp($duration_ep),
                    'duration' => $duration_ep,
                    'bitrate' => isset($ep['bitrate']) ? $ep['bitrate'] : 0,
                ],
                'custom_sid' => isset($ep['custom_sid']) ? $ep['custom_sid'] : '',
                'added' => isset($ep['added']) ? $ep['added'] : '',
                'season' => $season,
                'direct_source' => isset($ep['direct_source']) ? $ep['direct_source'] : ''
            ];
        }
        // Temporadas info
        $query_seasons = $conexao->prepare("SELECT * FROM series_seasons WHERE series_id = :id ORDER BY season_number ASC");
        $query_seasons->bindParam(':id', $series_id_int, PDO::PARAM_INT);
        $query_seasons->execute();
        $season_info_raw = $query_seasons->fetchAll(PDO::FETCH_ASSOC);
        $seasons_info = array_map(function($s) {
            $s['id'] = isset($s['id']) ? (string)$s['id'] : null;
            return $s;
        }, $season_info_raw);

        echo json_encode([
            'info' => $info,
            'episodes' => $episodes_grouped,
            'seasons' => $seasons_info
        ], JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT);
        exit();
    } catch (PDOException $e) {
        error_log("API XUI - Erro get_series_info: " . $e->getMessage());
        http_response_code(500);
        echo json_encode(['error' => 'Erro interno [get_series_info]']);
        exit();
    }
}

// Ação inválida
http_response_code(400);
error_log("API XUI - Ação inválida: $action para usuário $username");
echo json_encode(['error' => 'Ação inválida ou não especificada']);
exit();
?>